/**
 * Copyright &copy; 2015-2020 <a href="http://www.jeeplus.org/">JeePlus</a> All rights reserved.
 */
package com.jeeplus.modules.warehouse.storage.web;

import java.util.Date;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.ConstraintViolationException;

import com.jeeplus.modules.sys.utils.UserUtils;
import com.jeeplus.modules.warehouse.code.util.CodeUtil;
import com.jeeplus.modules.warehouse.code.util.StaticNumSeq;
import com.jeeplus.modules.warehouse.goods.entity.Goods;
import com.jeeplus.modules.warehouse.goods.entity.GoodsInfo;
import com.jeeplus.modules.warehouse.materialtype.entity.MaterialType;
import com.jeeplus.modules.warehouse.qrcode.entity.QrCode;
import com.jeeplus.modules.warehouse.storage.dto.CountStorageInfo;
import com.jeeplus.modules.warehouse.storage.entity.StorageInfo;
import org.apache.shiro.authz.annotation.Logical;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.multipart.MultipartFile;

import com.google.common.collect.Lists;
import com.jeeplus.common.utils.DateUtils;
import com.jeeplus.common.config.Global;
import com.jeeplus.common.json.AjaxJson;
import com.jeeplus.core.persistence.Page;
import com.jeeplus.core.web.BaseController;
import com.jeeplus.common.utils.StringUtils;
import com.jeeplus.common.utils.excel.ExportExcel;
import com.jeeplus.common.utils.excel.ImportExcel;
import com.jeeplus.modules.warehouse.storage.entity.Storage;
import com.jeeplus.modules.warehouse.storage.service.StorageService;

/**
 * 入库信息Controller
 * @author huyi
 * @version 2023-02-13
 */
@Controller
@RequestMapping(value = "${adminPath}/warehouse/storage/storage")
public class StorageController extends BaseController {

	@Autowired
	private StorageService storageService;
	
	@ModelAttribute
	public Storage get(@RequestParam(required=false) String id) {
		Storage entity = null;
		if (StringUtils.isNotBlank(id)){
			entity = storageService.get(id);
		}
		if (entity == null){
			entity = new Storage();
		}
		return entity;
	}
	
	/**
	 * 入库信息列表页面
	 */
	@RequiresPermissions("warehouse:storage:storage:list")
	@RequestMapping(value = {"list", ""})
	public String list(Storage storage, Model model) {
		model.addAttribute("storage", storage);
		return "modules/warehouse/storage/storageList";
	}
	
		/**
	 * 入库信息列表数据
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:storage:storage:list")
	@RequestMapping(value = "data")
	public Map<String, Object> data(Storage storage, HttpServletRequest request, HttpServletResponse response, Model model) {
		Page<Storage> page = storageService.findPage(new Page<Storage>(request, response), storage); 
		return getBootstrapData(page);
	}

	/**
	 * 查看，增加，编辑入库信息表单页面
	 */
	@RequiresPermissions(value={"warehouse:storage:storage:view","warehouse:storage:storage:add","warehouse:storage:storage:edit"},logical=Logical.OR)
	@RequestMapping(value = "form/{mode}")
	public String form(@PathVariable String mode, Storage storage, Model model) {
		String code = CodeUtil.code("RKD", StaticNumSeq.RKD);
		storage.setCode(code);
		storage.setTime(new Date());
		storage.setOperator(UserUtils.getUser());
		model.addAttribute("storage", storage);
		model.addAttribute("mode", mode);
		return "modules/warehouse/storage/storageForm";
	}

	/**
	 * 保存入库信息
	 */
	@ResponseBody
	@RequiresPermissions(value={"warehouse:storage:storage:add","warehouse:storage:storage:edit"},logical=Logical.OR)
	@RequestMapping(value = "save")
	public AjaxJson save(Storage storage, Model model) throws Exception{
		AjaxJson j = new AjaxJson();
		/**
		 * 后台hibernate-validation插件校验
		 */
		String errMsg = beanValidator(storage);
		if (StringUtils.isNotBlank(errMsg)){
			j.setSuccess(false);
			j.setMsg(errMsg);
			return j;
		}
		//新增或编辑表单保存
		storageService.save(storage);//保存
		j.setSuccess(true);
		j.setMsg("保存入库信息成功");
		return j;
	}
	
	/**
	 * 删除入库信息
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:storage:storage:del")
	@RequestMapping(value = "delete")
	public AjaxJson delete(Storage storage) {
		AjaxJson j = new AjaxJson();
		storageService.delete(storage);
		j.setMsg("删除入库信息成功");
		return j;
	}
	
	/**
	 * 批量删除入库信息
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:storage:storage:del")
	@RequestMapping(value = "deleteAll")
	public AjaxJson deleteAll(String ids) {
		AjaxJson j = new AjaxJson();
		String idArray[] =ids.split(",");
		for(String id : idArray){
			storageService.delete(storageService.get(id));
		}
		j.setMsg("删除入库信息成功");
		return j;
	}
	
	/**
	 * 导出excel文件
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:storage:storage:export")
    @RequestMapping(value = "export")
    public AjaxJson exportFile(Storage storage, HttpServletRequest request, HttpServletResponse response) {
		AjaxJson j = new AjaxJson();
		try {
            String fileName = "入库信息"+DateUtils.getDate("yyyyMMddHHmmss")+".xlsx";
            Page<Storage> page = storageService.findPage(new Page<Storage>(request, response, -1), storage);
    		new ExportExcel("入库信息", Storage.class).setDataList(page.getList()).write(response, fileName).dispose();
    		j.setSuccess(true);
    		j.setMsg("导出成功！");
    		return j;
		} catch (Exception e) {
			j.setSuccess(false);
			j.setMsg("导出入库信息记录失败！失败信息："+e.getMessage());
		}
			return j;
    }
    
    @ResponseBody
    @RequestMapping(value = "detail")
	public Storage detail(String id) {
		return storageService.get(id);
	}

	@ResponseBody
	@RequestMapping(value = "countStorage")
	public Storage countStorage(String id) {
		return storageService.countStorage(id);
	}

	@ResponseBody
	@RequestMapping(value = "findInfoBycountStorage")
	public AjaxJson findInfoBycountStorage(CountStorageInfo countStorageInfo) {
		AjaxJson j = new AjaxJson();
		List<StorageInfo> storageInfoList = storageService.findInfoBycountStorage(countStorageInfo);
		j.put("storageInfoList", storageInfoList);
		j.setSuccess(true);
		j.setMsg("查询成功");
		return j;
	}

	/**
	 * 导入Excel数据

	 */
	@ResponseBody
	@RequiresPermissions("warehouse:storage:storage:import")
    @RequestMapping(value = "import")
   	public AjaxJson importFile(@RequestParam("file")MultipartFile file, HttpServletResponse response, HttpServletRequest request) {
		AjaxJson j = new AjaxJson();
		try {
			int successNum = 0;
			int failureNum = 0;
			StringBuilder failureMsg = new StringBuilder();
			ImportExcel ei = new ImportExcel(file, 1, 0);
			List<Storage> list = ei.getDataList(Storage.class);
			for (Storage storage : list){
				try{
					storageService.save(storage);
					successNum++;
				}catch(ConstraintViolationException ex){
					failureNum++;
				}catch (Exception ex) {
					failureNum++;
				}
			}
			if (failureNum>0){
				failureMsg.insert(0, "，失败 "+failureNum+" 条入库信息记录。");
			}
			j.setMsg( "已成功导入 "+successNum+" 条入库信息记录"+failureMsg);
		} catch (Exception e) {
			j.setSuccess(false);
			j.setMsg("导入入库信息失败！失败信息："+e.getMessage());
		}
		return j;
    }
	
	/**
	 * 下载导入入库信息数据模板
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:storage:storage:import")
    @RequestMapping(value = "import/template")
     public AjaxJson importFileTemplate(HttpServletResponse response) {
		AjaxJson j = new AjaxJson();
		try {
            String fileName = "入库信息数据导入模板.xlsx";
    		List<Storage> list = Lists.newArrayList(); 
    		new ExportExcel("入库信息数据", Storage.class, 1).setDataList(list).write(response, fileName).dispose();
    		return null;
		} catch (Exception e) {
			j.setSuccess(false);
			j.setMsg( "导入模板下载失败！失败信息："+e.getMessage());
		}
		return j;
    }

	/**
	 * 查询需要入库的货品信息
	 * @param goodsInfo
	 * @return
	 */
	@ResponseBody
	@RequestMapping(value = "findByPcTypeModel")
	public AjaxJson findByPcTypeModel(GoodsInfo goodsInfo) {
		Goods goods = new Goods();
		goods.setBatchNum("PC-2023-02-0003");
		goodsInfo.setType(new MaterialType("3c56529c82bd49c38ebd27d5264af130"));
		goodsInfo.setGoods(goods);

		AjaxJson j = new AjaxJson();
		if (null != goodsInfo && null != goodsInfo.getGoods() && null != goodsInfo.getType() && StringUtils.isNotBlank(goodsInfo.getGoods().getBatchNum()) && StringUtils.isNotBlank(goodsInfo.getType().getId())) {
			List<StorageInfo> storageInfoList = storageService.findByPcTypeModel(goodsInfo);
			j.put("storageInfoList", storageInfoList);
			j.setSuccess(true);
			j.setMsg("查询成功");
		} else {
			j.setSuccess(false);
			j.setMsg("查询条件不能为空");
		}
		return j;
	}

}