/**
 * Copyright &copy; 2015-2020 <a href="http://www.jeeplus.org/">JeePlus</a> All rights reserved.
 */
package com.jeeplus.modules.warehouse.outbound.web;

import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.ConstraintViolationException;

import org.apache.shiro.authz.annotation.Logical;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.multipart.MultipartFile;

import com.google.common.collect.Lists;
import com.jeeplus.common.utils.DateUtils;
import com.jeeplus.common.config.Global;
import com.jeeplus.common.json.AjaxJson;
import com.jeeplus.core.persistence.Page;
import com.jeeplus.core.web.BaseController;
import com.jeeplus.common.utils.StringUtils;
import com.jeeplus.common.utils.excel.ExportExcel;
import com.jeeplus.common.utils.excel.ImportExcel;
import com.jeeplus.modules.warehouse.outbound.entity.Outbound;
import com.jeeplus.modules.warehouse.outbound.service.OutboundService;

/**
 * 出库信息Controller
 * @author zhanglt
 * @version 2023-02-14
 */
@Controller
@RequestMapping(value = "${adminPath}/warehouse/outbound/outbound")
public class OutboundController extends BaseController {

	@Autowired
	private OutboundService outboundService;
	
	@ModelAttribute
	public Outbound get(@RequestParam(required=false) String id) {
		Outbound entity = null;
		if (StringUtils.isNotBlank(id)){
			entity = outboundService.get(id);
		}
		if (entity == null){
			entity = new Outbound();
		}
		return entity;
	}
	
	/**
	 * 出库信息列表页面
	 */
	@RequiresPermissions("warehouse:outbound:outbound:list")
	@RequestMapping(value = {"list", ""})
	public String list(Outbound outbound, Model model) {
		model.addAttribute("outbound", outbound);
		return "modules/warehouse/outbound/outboundList";
	}
	
		/**
	 * 出库信息列表数据
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:outbound:outbound:list")
	@RequestMapping(value = "data")
	public Map<String, Object> data(Outbound outbound, HttpServletRequest request, HttpServletResponse response, Model model) {
		Page<Outbound> page = outboundService.findPage(new Page<Outbound>(request, response), outbound); 
		return getBootstrapData(page);
	}

	/**
	 * 查看，增加，编辑出库信息表单页面
	 */
	@RequiresPermissions(value={"warehouse:outbound:outbound:view","warehouse:outbound:outbound:add","warehouse:outbound:outbound:edit"},logical=Logical.OR)
	@RequestMapping(value = "form/{mode}")
	public String form(@PathVariable String mode, Outbound outbound, Model model) {
		model.addAttribute("outbound", outbound);
		model.addAttribute("mode", mode);
		return "modules/warehouse/outbound/outboundForm";
	}

	/**
	 * 保存出库信息
	 */
	@ResponseBody
	@RequiresPermissions(value={"warehouse:outbound:outbound:add","warehouse:outbound:outbound:edit"},logical=Logical.OR)
	@RequestMapping(value = "save")
	public AjaxJson save(Outbound outbound, Model model) throws Exception{
		AjaxJson j = new AjaxJson();
		/**
		 * 后台hibernate-validation插件校验
		 */
		String errMsg = beanValidator(outbound);
		if (StringUtils.isNotBlank(errMsg)){
			j.setSuccess(false);
			j.setMsg(errMsg);
			return j;
		}
		//新增或编辑表单保存
		outboundService.save(outbound);//保存
		j.setSuccess(true);
		j.setMsg("保存出库信息成功");
		return j;
	}
	
	/**
	 * 删除出库信息
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:outbound:outbound:del")
	@RequestMapping(value = "delete")
	public AjaxJson delete(Outbound outbound) {
		AjaxJson j = new AjaxJson();
		outboundService.delete(outbound);
		j.setMsg("删除出库信息成功");
		return j;
	}
	
	/**
	 * 批量删除出库信息
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:outbound:outbound:del")
	@RequestMapping(value = "deleteAll")
	public AjaxJson deleteAll(String ids) {
		AjaxJson j = new AjaxJson();
		String idArray[] =ids.split(",");
		for(String id : idArray){
			outboundService.delete(outboundService.get(id));
		}
		j.setMsg("删除出库信息成功");
		return j;
	}
	
	/**
	 * 导出excel文件
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:outbound:outbound:export")
    @RequestMapping(value = "export")
    public AjaxJson exportFile(Outbound outbound, HttpServletRequest request, HttpServletResponse response) {
		AjaxJson j = new AjaxJson();
		try {
            String fileName = "出库信息"+DateUtils.getDate("yyyyMMddHHmmss")+".xlsx";
            Page<Outbound> page = outboundService.findPage(new Page<Outbound>(request, response, -1), outbound);
    		new ExportExcel("出库信息", Outbound.class).setDataList(page.getList()).write(response, fileName).dispose();
    		j.setSuccess(true);
    		j.setMsg("导出成功！");
    		return j;
		} catch (Exception e) {
			j.setSuccess(false);
			j.setMsg("导出出库信息记录失败！失败信息："+e.getMessage());
		}
			return j;
    }
    
    @ResponseBody
    @RequestMapping(value = "detail")
	public Outbound detail(String id) {
		return outboundService.get(id);
	}
	

	/**
	 * 导入Excel数据

	 */
	@ResponseBody
	@RequiresPermissions("warehouse:outbound:outbound:import")
    @RequestMapping(value = "import")
   	public AjaxJson importFile(@RequestParam("file")MultipartFile file, HttpServletResponse response, HttpServletRequest request) {
		AjaxJson j = new AjaxJson();
		try {
			int successNum = 0;
			int failureNum = 0;
			StringBuilder failureMsg = new StringBuilder();
			ImportExcel ei = new ImportExcel(file, 1, 0);
			List<Outbound> list = ei.getDataList(Outbound.class);
			for (Outbound outbound : list){
				try{
					outboundService.save(outbound);
					successNum++;
				}catch(ConstraintViolationException ex){
					failureNum++;
				}catch (Exception ex) {
					failureNum++;
				}
			}
			if (failureNum>0){
				failureMsg.insert(0, "，失败 "+failureNum+" 条出库信息记录。");
			}
			j.setMsg( "已成功导入 "+successNum+" 条出库信息记录"+failureMsg);
		} catch (Exception e) {
			j.setSuccess(false);
			j.setMsg("导入出库信息失败！失败信息："+e.getMessage());
		}
		return j;
    }
	
	/**
	 * 下载导入出库信息数据模板
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:outbound:outbound:import")
    @RequestMapping(value = "import/template")
     public AjaxJson importFileTemplate(HttpServletResponse response) {
		AjaxJson j = new AjaxJson();
		try {
            String fileName = "出库信息数据导入模板.xlsx";
    		List<Outbound> list = Lists.newArrayList(); 
    		new ExportExcel("出库信息数据", Outbound.class, 1).setDataList(list).write(response, fileName).dispose();
    		return null;
		} catch (Exception e) {
			j.setSuccess(false);
			j.setMsg( "导入模板下载失败！失败信息："+e.getMessage());
		}
		return j;
    }
	

}