/**
 * Copyright &copy; 2015-2020 <a href="http://www.jeeplus.org/">JeePlus</a> All rights reserved.
 */
package com.jeeplus.modules.warehouse.ledger.web;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.ConstraintViolationException;

import org.apache.shiro.authz.annotation.Logical;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.multipart.MultipartFile;

import com.google.common.collect.Lists;
import com.jeeplus.common.utils.DateUtils;
import com.jeeplus.common.config.Global;
import com.jeeplus.common.json.AjaxJson;
import com.jeeplus.core.persistence.Page;
import com.jeeplus.core.web.BaseController;
import com.jeeplus.common.utils.StringUtils;
import com.jeeplus.common.utils.excel.ExportExcel;
import com.jeeplus.common.utils.excel.ImportExcel;
import com.jeeplus.modules.warehouse.ledger.entity.Ledger;
import com.jeeplus.modules.warehouse.ledger.service.LedgerService;

/**
 * 台账信息Controller
 * @author huyi
 * @version 2023-02-13
 */
@Controller
@RequestMapping(value = "${adminPath}/warehouse/ledger/ledger")
public class LedgerController extends BaseController {

	@Autowired
	private LedgerService ledgerService;
	
	@ModelAttribute
	public Ledger get(@RequestParam(required=false) String id) {
		Ledger entity = null;
		if (StringUtils.isNotBlank(id)){
			entity = ledgerService.get(id);
		}
		if (entity == null){
			entity = new Ledger();
		}
		return entity;
	}
	
	/**
	 * 台账信息列表页面
	 */
	@RequiresPermissions("warehouse:ledger:ledger:list")
	@RequestMapping(value = {"list", ""})
	public String list(Ledger ledger, Model model) {
		model.addAttribute("ledger", ledger);
		return "modules/warehouse/ledger/ledgerList";
	}
	
		/**
	 * 台账信息列表数据
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:ledger:ledger:list")
	@RequestMapping(value = "data")
	public Map<String, Object> data(Ledger ledger, HttpServletRequest request, HttpServletResponse response, Model model) {
		Page<Ledger> page = ledgerService.findPage(new Page<Ledger>(request, response), ledger); 
		return getBootstrapData(page);
	}

	/**
	 * 查看，增加，编辑台账信息表单页面
	 */
	@RequiresPermissions(value={"warehouse:ledger:ledger:view","warehouse:ledger:ledger:add","warehouse:ledger:ledger:edit"},logical=Logical.OR)
	@RequestMapping(value = "form/{mode}")
	public String form(@PathVariable String mode, Ledger ledger, Model model) {
		model.addAttribute("ledger", ledger);
		model.addAttribute("mode", mode);
		return "modules/warehouse/ledger/ledgerForm";
	}

	/**
	 * 保存台账信息
	 */
	@ResponseBody
	@RequiresPermissions(value={"warehouse:ledger:ledger:add","warehouse:ledger:ledger:edit"},logical=Logical.OR)
	@RequestMapping(value = "save")
	public AjaxJson save(Ledger ledger, Model model) throws Exception{
		AjaxJson j = new AjaxJson();
		/**
		 * 后台hibernate-validation插件校验
		 */
		String errMsg = beanValidator(ledger);
		if (StringUtils.isNotBlank(errMsg)){
			j.setSuccess(false);
			j.setMsg(errMsg);
			return j;
		}
		//新增或编辑表单保存
		ledgerService.save(ledger);//保存
		j.setSuccess(true);
		j.setMsg("保存台账信息成功");
		return j;
	}
	
	/**
	 * 删除台账信息
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:ledger:ledger:del")
	@RequestMapping(value = "delete")
	public AjaxJson delete(Ledger ledger) {
		AjaxJson j = new AjaxJson();
		ledgerService.delete(ledger);
		j.setMsg("删除台账信息成功");
		return j;
	}
	
	/**
	 * 批量删除台账信息
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:ledger:ledger:del")
	@RequestMapping(value = "deleteAll")
	public AjaxJson deleteAll(String ids) {
		AjaxJson j = new AjaxJson();
		String idArray[] =ids.split(",");
		for(String id : idArray){
			ledgerService.delete(ledgerService.get(id));
		}
		j.setMsg("删除台账信息成功");
		return j;
	}
	
	/**
	 * 导出excel文件
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:ledger:ledger:export")
    @RequestMapping(value = "export")
    public AjaxJson exportFile(Ledger ledger, HttpServletRequest request, HttpServletResponse response) {
		AjaxJson j = new AjaxJson();
		try {
            String fileName = "台账信息"+DateUtils.getDate("yyyyMMddHHmmss")+".xlsx";
            Page<Ledger> page = ledgerService.findPage(new Page<Ledger>(request, response, -1), ledger);
    		new ExportExcel("台账信息", Ledger.class).setDataList(page.getList()).write(response, fileName).dispose();
    		j.setSuccess(true);
    		j.setMsg("导出成功！");
    		return j;
		} catch (Exception e) {
			j.setSuccess(false);
			j.setMsg("导出台账信息记录失败！失败信息："+e.getMessage());
		}
			return j;
    }
    
    @ResponseBody
    @RequestMapping(value = "detail")
	public Ledger detail(String id) {
		return ledgerService.get(id);
	}

	public Ledger countLedger(String id) {
		return ledgerService.countLedger(id);
	}

	/**
	 * 导入Excel数据

	 */
	@ResponseBody
	@RequiresPermissions("warehouse:ledger:ledger:import")
    @RequestMapping(value = "import")
   	public AjaxJson importFile(@RequestParam("file")MultipartFile file, HttpServletResponse response, HttpServletRequest request) {
		AjaxJson j = new AjaxJson();
		try {
			int successNum = 0;
			int failureNum = 0;
			StringBuilder failureMsg = new StringBuilder();
			ImportExcel ei = new ImportExcel(file, 1, 0);
			List<Ledger> list = ei.getDataList(Ledger.class);
			for (Ledger ledger : list){
				try{
					ledgerService.save(ledger);
					successNum++;
				}catch(ConstraintViolationException ex){
					failureNum++;
				}catch (Exception ex) {
					failureNum++;
				}
			}
			if (failureNum>0){
				failureMsg.insert(0, "，失败 "+failureNum+" 条台账信息记录。");
			}
			j.setMsg( "已成功导入 "+successNum+" 条台账信息记录"+failureMsg);
		} catch (Exception e) {
			j.setSuccess(false);
			j.setMsg("导入台账信息失败！失败信息："+e.getMessage());
		}
		return j;
    }
	
	/**
	 * 下载导入台账信息数据模板
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:ledger:ledger:import")
    @RequestMapping(value = "import/template")
     public AjaxJson importFileTemplate(HttpServletResponse response) {
		AjaxJson j = new AjaxJson();
		try {
            String fileName = "台账信息数据导入模板.xlsx";
    		List<Ledger> list = Lists.newArrayList(); 
    		new ExportExcel("台账信息数据", Ledger.class, 1).setDataList(list).write(response, fileName).dispose();
    		return null;
		} catch (Exception e) {
			j.setSuccess(false);
			j.setMsg( "导入模板下载失败！失败信息："+e.getMessage());
		}
		return j;
    }
	

}