/**
 * Copyright &copy; 2015-2020 <a href="http://www.jeeplus.org/">JeePlus</a> All rights reserved.
 */
package com.jeeplus.modules.warehouse.materialloss.web;

import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.ConstraintViolationException;

import com.jeeplus.modules.sys.utils.UserUtils;
import org.apache.shiro.authz.annotation.Logical;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.multipart.MultipartFile;

import com.google.common.collect.Lists;
import com.jeeplus.common.utils.DateUtils;
import com.jeeplus.common.config.Global;
import com.jeeplus.common.json.AjaxJson;
import com.jeeplus.core.persistence.Page;
import com.jeeplus.core.web.BaseController;
import com.jeeplus.common.utils.StringUtils;
import com.jeeplus.common.utils.excel.ExportExcel;
import com.jeeplus.common.utils.excel.ImportExcel;
import com.jeeplus.modules.warehouse.materialloss.entity.MaterialLoss;
import com.jeeplus.modules.warehouse.materialloss.service.MaterialLossService;

/**
 * 物资盘亏记录Controller
 * @author zhanglt
 * @version 2023-02-14
 */
@Controller
@RequestMapping(value = "${adminPath}/warehouse/materialloss/materialLoss")
public class MaterialLossController extends BaseController {

	@Autowired
	private MaterialLossService materialLossService;
	
	@ModelAttribute
	public MaterialLoss get(@RequestParam(required=false) String id) {
		MaterialLoss entity = null;
		if (StringUtils.isNotBlank(id)){
			entity = materialLossService.get(id);
		}
		if (entity == null){
			entity = new MaterialLoss();
		}
		return entity;
	}
	
	/**
	 * 物资盘亏记录列表页面
	 */
	@RequiresPermissions("warehouse:materialloss:materialLoss:list")
	@RequestMapping(value = {"list", ""})
	public String list(MaterialLoss materialLoss, Model model) {
		model.addAttribute("materialLoss", materialLoss);
		return "modules/warehouse/materialloss/materialLossList";
	}
	
		/**
	 * 物资盘亏记录列表数据
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:materialloss:materialLoss:list")
	@RequestMapping(value = "data")
	public Map<String, Object> data(MaterialLoss materialLoss, HttpServletRequest request, HttpServletResponse response, Model model) {
		Page<MaterialLoss> page = materialLossService.findPage(new Page<MaterialLoss>(request, response), materialLoss); 
		return getBootstrapData(page);
	}

	/**
	 * 查看，增加，编辑物资盘亏记录表单页面
	 */
	@RequiresPermissions(value={"warehouse:materialloss:materialLoss:view","warehouse:materialloss:materialLoss:add","warehouse:materialloss:materialLoss:edit"},logical=Logical.OR)
	@RequestMapping(value = "form/{mode}")
	public String form(@PathVariable String mode, MaterialLoss materialLoss, Model model) {
		if(mode != null && mode.equals("add")){
			if(Objects.equals(materialLoss.getNumber(), "")||materialLoss.getNumber()==null){
				materialLoss.setNumber(materialLossService.createPKDH());
			}
			if(materialLoss.getOperator() == null){
				materialLoss.setOperator(UserUtils.getUser());
			}
		}
		model.addAttribute("materialLoss", materialLoss);
		model.addAttribute("mode", mode);
		return "modules/warehouse/materialloss/materialLossForm";
	}

	/**
	 * 保存物资盘亏记录
	 */
	@ResponseBody
	@RequiresPermissions(value={"warehouse:materialloss:materialLoss:add","warehouse:materialloss:materialLoss:edit"},logical=Logical.OR)
	@RequestMapping(value = "save")
	public AjaxJson save(MaterialLoss materialLoss, Model model) throws Exception{
		AjaxJson j = new AjaxJson();
		/**
		 * 后台hibernate-validation插件校验
		 */
		String errMsg = beanValidator(materialLoss);
		if (StringUtils.isNotBlank(errMsg)){
			j.setSuccess(false);
			j.setMsg(errMsg);
			return j;
		}
		//新增或编辑表单保存
//		materialLossService.save(materialLoss);//保存
		materialLossService.delivery(materialLoss);//保存
		j.setSuccess(true);
		j.setMsg("保存物资盘亏记录成功");
		return j;
	}
	
	/**
	 * 删除物资盘亏记录
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:materialloss:materialLoss:del")
	@RequestMapping(value = "delete")
	public AjaxJson delete(MaterialLoss materialLoss) {
		AjaxJson j = new AjaxJson();
		materialLossService.delete(materialLoss);
		j.setMsg("删除物资盘亏记录成功");
		return j;
	}
	
	/**
	 * 批量删除物资盘亏记录
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:materialloss:materialLoss:del")
	@RequestMapping(value = "deleteAll")
	public AjaxJson deleteAll(String ids) {
		AjaxJson j = new AjaxJson();
		String idArray[] =ids.split(",");
		for(String id : idArray){
			materialLossService.delete(materialLossService.get(id));
		}
		j.setMsg("删除物资盘亏记录成功");
		return j;
	}
	
	/**
	 * 导出excel文件
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:materialloss:materialLoss:export")
    @RequestMapping(value = "export")
    public AjaxJson exportFile(MaterialLoss materialLoss, HttpServletRequest request, HttpServletResponse response) {
		AjaxJson j = new AjaxJson();
		try {
            String fileName = "物资盘亏记录"+DateUtils.getDate("yyyyMMddHHmmss")+".xlsx";
            Page<MaterialLoss> page = materialLossService.findPage(new Page<MaterialLoss>(request, response, -1), materialLoss);
    		new ExportExcel("物资盘亏记录", MaterialLoss.class).setDataList(page.getList()).write(response, fileName).dispose();
    		j.setSuccess(true);
    		j.setMsg("导出成功！");
    		return j;
		} catch (Exception e) {
			j.setSuccess(false);
			j.setMsg("导出物资盘亏记录记录失败！失败信息："+e.getMessage());
		}
			return j;
    }
    
    @ResponseBody
    @RequestMapping(value = "detail")
	public MaterialLoss detail(String id) {
		return materialLossService.get(id);
	}
	

	/**
	 * 导入Excel数据

	 */
	@ResponseBody
	@RequiresPermissions("warehouse:materialloss:materialLoss:import")
    @RequestMapping(value = "import")
   	public AjaxJson importFile(@RequestParam("file")MultipartFile file, HttpServletResponse response, HttpServletRequest request) {
		AjaxJson j = new AjaxJson();
		try {
			int successNum = 0;
			int failureNum = 0;
			StringBuilder failureMsg = new StringBuilder();
			ImportExcel ei = new ImportExcel(file, 1, 0);
			List<MaterialLoss> list = ei.getDataList(MaterialLoss.class);
			for (MaterialLoss materialLoss : list){
				try{
					materialLossService.save(materialLoss);
					successNum++;
				}catch(ConstraintViolationException ex){
					failureNum++;
				}catch (Exception ex) {
					failureNum++;
				}
			}
			if (failureNum>0){
				failureMsg.insert(0, "，失败 "+failureNum+" 条物资盘亏记录记录。");
			}
			j.setMsg( "已成功导入 "+successNum+" 条物资盘亏记录记录"+failureMsg);
		} catch (Exception e) {
			j.setSuccess(false);
			j.setMsg("导入物资盘亏记录失败！失败信息："+e.getMessage());
		}
		return j;
    }
	
	/**
	 * 下载导入物资盘亏记录数据模板
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:materialloss:materialLoss:import")
    @RequestMapping(value = "import/template")
     public AjaxJson importFileTemplate(HttpServletResponse response) {
		AjaxJson j = new AjaxJson();
		try {
            String fileName = "物资盘亏记录数据导入模板.xlsx";
    		List<MaterialLoss> list = Lists.newArrayList(); 
    		new ExportExcel("物资盘亏记录数据", MaterialLoss.class, 1).setDataList(list).write(response, fileName).dispose();
    		return null;
		} catch (Exception e) {
			j.setSuccess(false);
			j.setMsg( "导入模板下载失败！失败信息："+e.getMessage());
		}
		return j;
    }

	/**
	 * 保存物资盘亏记录
	 */
	@ResponseBody
	@RequiresPermissions(value={"warehouse:materialloss:materialLoss:add","warehouse:materialloss:materialLoss:edit"},logical=Logical.OR)
	@RequestMapping(value = "delivery")
	public AjaxJson delivery(MaterialLoss materialLoss, Model model) throws Exception{
		AjaxJson j = new AjaxJson();
		/**
		 * 后台hibernate-validation插件校验
		 */
		String errMsg = beanValidator(materialLoss);
		if (StringUtils.isNotBlank(errMsg)){
			j.setSuccess(false);
			j.setMsg(errMsg);
			return j;
		}
		//新增或编辑表单保存
//		materialLossService.save(materialLoss);//保存
		materialLossService.delivery(materialLoss);//保存
		j.setSuccess(true);
		j.setMsg("保存物资盘亏记录成功");
		return j;
	}
}