/**
 * Copyright &copy; 2015-2020 <a href="http://www.jeeplus.org/">JeePlus</a> All rights reserved.
 */
package com.jeeplus.modules.warehouse.shelves.web;

import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.ConstraintViolationException;

import com.jeeplus.modules.warehouse.warehouse.entity.Warehouse;
import com.jeeplus.modules.warehouse.warehouse.service.WarehouseService;
import org.apache.shiro.authz.annotation.Logical;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.multipart.MultipartFile;

import com.google.common.collect.Lists;
import com.jeeplus.common.utils.DateUtils;
import com.jeeplus.common.config.Global;
import com.jeeplus.common.json.AjaxJson;
import com.jeeplus.core.persistence.Page;
import com.jeeplus.core.web.BaseController;
import com.jeeplus.common.utils.StringUtils;
import com.jeeplus.common.utils.excel.ExportExcel;
import com.jeeplus.common.utils.excel.ImportExcel;
import com.jeeplus.modules.warehouse.shelves.entity.Shelves;
import com.jeeplus.modules.warehouse.shelves.service.ShelvesService;

/**
 * 货架信息Controller
 * @author zhanglt
 * @version 2023-02-13
 */
@Controller
@RequestMapping(value = "${adminPath}/warehouse/shelves/shelves")
public class ShelvesController extends BaseController {

	@Autowired
	private ShelvesService shelvesService;
	@Autowired
	private WarehouseService warehouseService;

	@ModelAttribute
	public Shelves get(@RequestParam(required=false) String id) {
		Shelves entity = null;
		if (StringUtils.isNotBlank(id)){
			entity = shelvesService.get(id);
		}
		if (entity == null){
			entity = new Shelves();
		}
		return entity;
	}
	
	/**
	 * 货架信息列表页面
	 */
	@RequiresPermissions("warehouse:shelves:shelves:list")
	@RequestMapping(value = {"list", ""})
	public String list(Shelves shelves, Model model) {
		model.addAttribute("shelves", shelves);
		Warehouse warehouse = new Warehouse();
		List<Warehouse> warehouseList = warehouseService.findList(warehouse);
		model.addAttribute("warehouseList", warehouseList);
		return "modules/warehouse/shelves/shelvesList";
	}
	
	/**
	 * 货架信息列表数据
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:shelves:shelves:list")
	@RequestMapping(value = "data")
	public Map<String, Object> data(Shelves shelves, HttpServletRequest request, HttpServletResponse response, Model model) {
		Page<Shelves> page = shelvesService.findPage(new Page<Shelves>(request, response), shelves); 
		return getBootstrapData(page);
	}

	/**
	 * 查看，增加，编辑货架信息表单页面
	 */
	@RequiresPermissions(value={"warehouse:shelves:shelves:view","warehouse:shelves:shelves:add","warehouse:shelves:shelves:edit"},logical=Logical.OR)
	@RequestMapping(value = "form/{mode}")
	public String form(@PathVariable String mode, Shelves shelves, Model model) {
		model.addAttribute("shelves", shelves);
		Warehouse warehouse = new Warehouse();
		List<Warehouse> warehouseList = warehouseService.findList(warehouse);
		model.addAttribute("warehouseList", warehouseList);
		model.addAttribute("mode", mode);
		return "modules/warehouse/shelves/shelvesForm";
	}

	/**
	 * 保存货架信息
	 */
	@ResponseBody
	@RequiresPermissions(value={"warehouse:shelves:shelves:add","warehouse:shelves:shelves:edit"},logical=Logical.OR)
	@RequestMapping(value = "save")
	public AjaxJson save(Shelves shelves, Model model) throws Exception{
		AjaxJson j = new AjaxJson();
		/**
		 * 后台hibernate-validation插件校验
		 */
		String errMsg = beanValidator(shelves);
		if (StringUtils.isNotBlank(errMsg)){
			j.setSuccess(false);
			j.setMsg(errMsg);
			return j;
		}
		//新增或编辑表单保存
		shelvesService.save(shelves);//保存
		j.setSuccess(true);
		j.setMsg("保存货架信息成功");
		return j;
	}
	
	/**
	 * 删除货架信息
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:shelves:shelves:del")
	@RequestMapping(value = "delete")
	public AjaxJson delete(Shelves shelves) {
		AjaxJson j = new AjaxJson();
		shelvesService.delete(shelves);
		j.setMsg("删除货架信息成功");
		return j;
	}
	
	/**
	 * 批量删除货架信息
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:shelves:shelves:del")
	@RequestMapping(value = "deleteAll")
	public AjaxJson deleteAll(String ids) {
		AjaxJson j = new AjaxJson();
		String idArray[] =ids.split(",");
		for(String id : idArray){
			shelvesService.delete(shelvesService.get(id));
		}
		j.setMsg("删除货架信息成功");
		return j;
	}
	
	/**
	 * 导出excel文件
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:shelves:shelves:export")
    @RequestMapping(value = "export")
    public AjaxJson exportFile(Shelves shelves, HttpServletRequest request, HttpServletResponse response) {
		AjaxJson j = new AjaxJson();
		try {
            String fileName = "货架信息"+DateUtils.getDate("yyyyMMddHHmmss")+".xlsx";
            Page<Shelves> page = shelvesService.findPage(new Page<Shelves>(request, response, -1), shelves);
    		new ExportExcel("货架信息", Shelves.class).setDataList(page.getList()).write(response, fileName).dispose();
    		j.setSuccess(true);
    		j.setMsg("导出成功！");
    		return j;
		} catch (Exception e) {
			j.setSuccess(false);
			j.setMsg("导出货架信息记录失败！失败信息："+e.getMessage());
		}
			return j;
    }

	/**
	 * 导入Excel数据

	 */
	@ResponseBody
	@RequiresPermissions("warehouse:shelves:shelves:import")
    @RequestMapping(value = "import")
   	public AjaxJson importFile(@RequestParam("file")MultipartFile file, HttpServletResponse response, HttpServletRequest request) {
		AjaxJson j = new AjaxJson();
		try {
			int successNum = 0;
			int failureNum = 0;
			StringBuilder failureMsg = new StringBuilder();
			ImportExcel ei = new ImportExcel(file, 1, 0);
			List<Shelves> list = ei.getDataList(Shelves.class);
			for (Shelves shelves : list){
				try{
					shelvesService.save(shelves);
					successNum++;
				}catch(ConstraintViolationException ex){
					failureNum++;
				}catch (Exception ex) {
					failureNum++;
				}
			}
			if (failureNum>0){
				failureMsg.insert(0, "，失败 "+failureNum+" 条货架信息记录。");
			}
			j.setMsg( "已成功导入 "+successNum+" 条货架信息记录"+failureMsg);
		} catch (Exception e) {
			j.setSuccess(false);
			j.setMsg("导入货架信息失败！失败信息："+e.getMessage());
		}
		return j;
    }
	
	/**
	 * 下载导入货架信息数据模板
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:shelves:shelves:import")
    @RequestMapping(value = "import/template")
     public AjaxJson importFileTemplate(HttpServletResponse response) {
		AjaxJson j = new AjaxJson();
		try {
            String fileName = "货架信息数据导入模板.xlsx";
    		List<Shelves> list = Lists.newArrayList(); 
    		new ExportExcel("货架信息数据", Shelves.class, 1).setDataList(list).write(response, fileName).dispose();
    		return null;
		} catch (Exception e) {
			j.setSuccess(false);
			j.setMsg( "导入模板下载失败！失败信息："+e.getMessage());
		}
		return j;
    }
	/**
	 * 货架信息列表页面
	 */
	@RequestMapping(value = "shelvesSelect")
	public String shelvesSelect(HttpServletRequest request, Shelves shelves, Model model) {
		model.addAttribute("shelves", shelves);
		model.addAttribute("isMultiSelect", request.getParameter("isMultiSelected"));
		return "modules/common/shelvesSelect";
	}
}
