/**
 * Copyright &copy; 2015-2020 <a href="http://www.jeeplus.org/">JeePlus</a> All rights reserved.
 */
package com.jeeplus.modules.warehouse.qrcode.web;

import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.ConstraintViolationException;

import org.apache.shiro.authz.annotation.Logical;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.multipart.MultipartFile;

import com.google.common.collect.Lists;
import com.jeeplus.common.utils.DateUtils;
import com.jeeplus.common.config.Global;
import com.jeeplus.common.json.AjaxJson;
import com.jeeplus.core.persistence.Page;
import com.jeeplus.core.web.BaseController;
import com.jeeplus.common.utils.StringUtils;
import com.jeeplus.common.utils.excel.ExportExcel;
import com.jeeplus.common.utils.excel.ImportExcel;
import com.jeeplus.modules.warehouse.qrcode.entity.QrCode;
import com.jeeplus.modules.warehouse.qrcode.service.QrCodeService;

/**
 * 二维码Controller
 * @author 胡懿
 * @version 2023-02-13
 */
@Controller
@RequestMapping(value = "${adminPath}/warehouse/qrcode/qrCode")
public class QrCodeController extends BaseController {

	@Autowired
	private QrCodeService qrCodeService;
	
	@ModelAttribute
	public QrCode get(@RequestParam(required=false) String id) {
		QrCode entity = null;
		if (StringUtils.isNotBlank(id)){
			entity = qrCodeService.get(id);
		}
		if (entity == null){
			entity = new QrCode();
		}
		return entity;
	}
	
	/**
	 * 二维码列表页面
	 */
	@RequiresPermissions("warehouse:qrcode:qrCode:list")
	@RequestMapping(value = {"list", ""})
	public String list(QrCode qrCode, Model model) {
		model.addAttribute("qrCode", qrCode);
		return "modules/warehouse/qrcode/qrCodeList";
	}
	
		/**
	 * 二维码列表数据
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:qrcode:qrCode:list")
	@RequestMapping(value = "data")
	public Map<String, Object> data(QrCode qrCode, HttpServletRequest request, HttpServletResponse response, Model model) {
		Page<QrCode> page = qrCodeService.findPage(new Page<QrCode>(request, response), qrCode); 
		return getBootstrapData(page);
	}

	/**
	 * 查看，增加，编辑二维码表单页面
	 */
	@RequiresPermissions(value={"warehouse:qrcode:qrCode:view","warehouse:qrcode:qrCode:add","warehouse:qrcode:qrCode:edit"},logical=Logical.OR)
	@RequestMapping(value = "form/{mode}")
	public String form(@PathVariable String mode, QrCode qrCode, Model model) {
		model.addAttribute("qrCode", qrCode);
		model.addAttribute("mode", mode);
		return "modules/warehouse/qrcode/qrCodeForm";
	}

	/**
	 * 保存二维码
	 */
	@ResponseBody
	@RequiresPermissions(value={"warehouse:qrcode:qrCode:add","warehouse:qrcode:qrCode:edit"},logical=Logical.OR)
	@RequestMapping(value = "save")
	public AjaxJson save(QrCode qrCode, Model model) throws Exception{
		AjaxJson j = new AjaxJson();
		/**
		 * 后台hibernate-validation插件校验
		 */
		String errMsg = beanValidator(qrCode);
		if (StringUtils.isNotBlank(errMsg)){
			j.setSuccess(false);
			j.setMsg(errMsg);
			return j;
		}
		//新增或编辑表单保存
		qrCodeService.save(qrCode);//保存
		j.setSuccess(true);
		j.setMsg("保存二维码成功");
		return j;
	}
	
	/**
	 * 删除二维码
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:qrcode:qrCode:del")
	@RequestMapping(value = "delete")
	public AjaxJson delete(QrCode qrCode) {
		AjaxJson j = new AjaxJson();
		qrCodeService.delete(qrCode);
		j.setMsg("删除二维码成功");
		return j;
	}
	
	/**
	 * 批量删除二维码
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:qrcode:qrCode:del")
	@RequestMapping(value = "deleteAll")
	public AjaxJson deleteAll(String ids) {
		AjaxJson j = new AjaxJson();
		String idArray[] =ids.split(",");
		for(String id : idArray){
			qrCodeService.delete(qrCodeService.get(id));
		}
		j.setMsg("删除二维码成功");
		return j;
	}
	
	/**
	 * 导出excel文件
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:qrcode:qrCode:export")
    @RequestMapping(value = "export")
    public AjaxJson exportFile(QrCode qrCode, HttpServletRequest request, HttpServletResponse response) {
		AjaxJson j = new AjaxJson();
		try {
            String fileName = "二维码"+DateUtils.getDate("yyyyMMddHHmmss")+".xlsx";
            Page<QrCode> page = qrCodeService.findPage(new Page<QrCode>(request, response, -1), qrCode);
    		new ExportExcel("二维码", QrCode.class).setDataList(page.getList()).write(response, fileName).dispose();
    		j.setSuccess(true);
    		j.setMsg("导出成功！");
    		return j;
		} catch (Exception e) {
			j.setSuccess(false);
			j.setMsg("导出二维码记录失败！失败信息："+e.getMessage());
		}
			return j;
    }

	/**
	 * 导入Excel数据

	 */
	@ResponseBody
	@RequiresPermissions("warehouse:qrcode:qrCode:import")
    @RequestMapping(value = "import")
   	public AjaxJson importFile(@RequestParam("file")MultipartFile file, HttpServletResponse response, HttpServletRequest request) {
		AjaxJson j = new AjaxJson();
		try {
			int successNum = 0;
			int failureNum = 0;
			StringBuilder failureMsg = new StringBuilder();
			ImportExcel ei = new ImportExcel(file, 1, 0);
			List<QrCode> list = ei.getDataList(QrCode.class);
			for (QrCode qrCode : list){
				try{
					qrCodeService.save(qrCode);
					successNum++;
				}catch(ConstraintViolationException ex){
					failureNum++;
				}catch (Exception ex) {
					failureNum++;
				}
			}
			if (failureNum>0){
				failureMsg.insert(0, "，失败 "+failureNum+" 条二维码记录。");
			}
			j.setMsg( "已成功导入 "+successNum+" 条二维码记录"+failureMsg);
		} catch (Exception e) {
			j.setSuccess(false);
			j.setMsg("导入二维码失败！失败信息："+e.getMessage());
		}
		return j;
    }
	
	/**
	 * 下载导入二维码数据模板
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:qrcode:qrCode:import")
    @RequestMapping(value = "import/template")
     public AjaxJson importFileTemplate(HttpServletResponse response) {
		AjaxJson j = new AjaxJson();
		try {
            String fileName = "二维码数据导入模板.xlsx";
    		List<QrCode> list = Lists.newArrayList(); 
    		new ExportExcel("二维码数据", QrCode.class, 1).setDataList(list).write(response, fileName).dispose();
    		return null;
		} catch (Exception e) {
			j.setSuccess(false);
			j.setMsg( "导入模板下载失败！失败信息："+e.getMessage());
		}
		return j;
    }

}