/**
 * Copyright &copy; 2015-2020 <a href="http://www.jeeplus.org/">JeePlus</a> All rights reserved.
 */
package com.jeeplus.modules.warehouse.goods.web;

import java.io.File;
import java.util.Date;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.ConstraintViolationException;

import com.jeeplus.common.utils.io.FileUtil;
import com.jeeplus.modules.sys.utils.UserUtils;
import com.jeeplus.modules.warehouse.code.util.CodeUtil;
import com.jeeplus.modules.warehouse.code.util.StaticNumSeq;
import com.jeeplus.modules.warehouse.goods.entity.GoodsInfo;
import com.jeeplus.modules.warehouse.materialtype.entity.MaterialType;
import com.jeeplus.modules.warehouse.qrcode.entity.QrCode;
import org.apache.shiro.authz.annotation.Logical;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.multipart.MultipartFile;

import com.google.common.collect.Lists;
import com.jeeplus.common.utils.DateUtils;
import com.jeeplus.common.config.Global;
import com.jeeplus.common.json.AjaxJson;
import com.jeeplus.core.persistence.Page;
import com.jeeplus.core.web.BaseController;
import com.jeeplus.common.utils.StringUtils;
import com.jeeplus.common.utils.excel.ExportExcel;
import com.jeeplus.common.utils.excel.ImportExcel;
import com.jeeplus.modules.warehouse.goods.entity.Goods;
import com.jeeplus.modules.warehouse.goods.service.GoodsService;

/**
 * 货品模块Controller
 * @author huyi
 * @version 2023-02-13
 */
@Controller
@RequestMapping(value = "${adminPath}/warehouse/goods/goods")
public class GoodsController extends BaseController {

	@Autowired
	private GoodsService goodsService;
	
	@ModelAttribute
	public Goods get(@RequestParam(required=false) String id) {
		Goods entity = null;
		if (StringUtils.isNotBlank(id)){
			entity = goodsService.get(id);
		}
		if (entity == null){
			entity = new Goods();
		}
		return entity;
	}
	
	/**
	 * 货品模块列表页面
	 */
	@RequiresPermissions("warehouse:goods:goods:list")
	@RequestMapping(value = {"list", ""})
	public String list(Goods goods, Model model) {
		model.addAttribute("goods", goods);
		return "modules/warehouse/goods/goodsList";
	}
	
		/**
	 * 货品模块列表数据
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:goods:goods:list")
	@RequestMapping(value = "data")
	public Map<String, Object> data(Goods goods, HttpServletRequest request, HttpServletResponse response, Model model) {
		Page<Goods> page = goodsService.findPage(new Page<Goods>(request, response), goods); 
		return getBootstrapData(page);
	}

	/**
	 * 查看，增加，编辑货品模块表单页面
	 */
	@RequiresPermissions(value={"warehouse:goods:goods:view","warehouse:goods:goods:add","warehouse:goods:goods:edit"},logical=Logical.OR)
	@RequestMapping(value = "form/{mode}")
	public String form(@PathVariable String mode, Goods goods, Model model) {
		/*if (mode.equals("add")) {
			String batchNum = CodeUtil.code("PC", StaticNumSeq.PC);
			goods.setBatchNum(batchNum);
		}*/

		goods.setTime(new Date());
		goods.setOperator(UserUtils.getUser());
		model.addAttribute("goods", goods);
		model.addAttribute("mode", mode);
		return "modules/warehouse/goods/goodsForm";
	}

	/**
	 * 保存货品模块
	 */
	@ResponseBody
	@RequiresPermissions(value={"warehouse:goods:goods:add","warehouse:goods:goods:edit"},logical=Logical.OR)
	@RequestMapping(value = "save")
	public AjaxJson save(Goods goods, Model model) throws Exception{
		AjaxJson j = new AjaxJson();
		/**
		 * 后台hibernate-validation插件校验
		 */
		String errMsg = beanValidator(goods);
		if (StringUtils.isNotBlank(errMsg)){
			j.setSuccess(false);
			j.setMsg(errMsg);
			return j;
		}
		//新增或编辑表单保存
		goodsService.save(goods);//保存
		j.setSuccess(true);
		j.setMsg("保存货品模块成功");
		return j;
	}
	
	/**
	 * 删除货品模块
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:goods:goods:del")
	@RequestMapping(value = "delete")
	public AjaxJson delete(Goods goods) {
		AjaxJson j = new AjaxJson();
		goodsService.delete(goods);
		j.setMsg("删除货品模块成功");
		return j;
	}
	
	/**
	 * 批量删除货品模块
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:goods:goods:del")
	@RequestMapping(value = "deleteAll")
	public AjaxJson deleteAll(String ids) {
		AjaxJson j = new AjaxJson();
		String idArray[] =ids.split(",");
		for(String id : idArray){
			goodsService.delete(goodsService.get(id));
		}
		j.setMsg("删除货品模块成功");
		return j;
	}
	
	/**
	 * 导出excel文件
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:goods:goods:export")
    @RequestMapping(value = "export")
    public AjaxJson exportFile(Goods goods, HttpServletRequest request, HttpServletResponse response) {
		AjaxJson j = new AjaxJson();
		try {
            String fileName = "货品模块"+DateUtils.getDate("yyyyMMddHHmmss")+".xlsx";
            Page<Goods> page = goodsService.findPage(new Page<Goods>(request, response, -1), goods);
    		new ExportExcel("货品模块", Goods.class).setDataList(page.getList()).write(response, fileName).dispose();
    		j.setSuccess(true);
    		j.setMsg("导出成功！");
    		return j;
		} catch (Exception e) {
			j.setSuccess(false);
			j.setMsg("导出货品模块记录失败！失败信息："+e.getMessage());
		}
			return j;
    }
    
    @ResponseBody
    @RequestMapping(value = "detail")
	public Goods detail(String id) {
		return goodsService.get(id);
	}
	

	/**
	 * 导入Excel数据

	 */
	@ResponseBody
	@RequiresPermissions("warehouse:goods:goods:import")
    @RequestMapping(value = "import")
   	public AjaxJson importFile(@RequestParam("file")MultipartFile file, HttpServletResponse response, HttpServletRequest request) {
		AjaxJson j = new AjaxJson();
		try {
			int successNum = 0;
			int failureNum = 0;
			StringBuilder failureMsg = new StringBuilder();
			ImportExcel ei = new ImportExcel(file, 1, 0);
			List<Goods> list = ei.getDataList(Goods.class);
			for (Goods goods : list){
				try{
					goodsService.save(goods);
					successNum++;
				}catch(ConstraintViolationException ex){
					failureNum++;
				}catch (Exception ex) {
					failureNum++;
				}
			}
			if (failureNum>0){
				failureMsg.insert(0, "，失败 "+failureNum+" 条货品模块记录。");
			}
			j.setMsg( "已成功导入 "+successNum+" 条货品模块记录"+failureMsg);
		} catch (Exception e) {
			j.setSuccess(false);
			j.setMsg("导入货品模块失败！失败信息："+e.getMessage());
		}
		return j;
    }
	
	/**
	 * 下载导入货品模块数据模板
	 */
	@ResponseBody
	@RequiresPermissions("warehouse:goods:goods:import")
    @RequestMapping(value = "import/template")
     public AjaxJson importFileTemplate(HttpServletResponse response) {
		AjaxJson j = new AjaxJson();
		try {
            String fileName = "货品模块数据导入模板.xlsx";
    		List<Goods> list = Lists.newArrayList(); 
    		new ExportExcel("货品模块数据", Goods.class, 1).setDataList(list).write(response, fileName).dispose();
    		return null;
		} catch (Exception e) {
			j.setSuccess(false);
			j.setMsg( "导入模板下载失败！失败信息："+e.getMessage());
		}
		return j;
    }

	/**
	 * 打印二维码
	 */
	@ResponseBody
	@RequestMapping(value = "printQr")
	public AjaxJson printQr(String goodsInfoId) throws Exception{
		AjaxJson j = new AjaxJson();
		//新增或编辑表单保存
		GoodsInfo goodsInfo = goodsService.printQr(goodsInfoId);//保存
		j.put("goodsInfo", goodsInfo);
		j.setSuccess(true);
		j.setMsg("保存货品模块成功");
		return j;
	}

	/**
	 * 清除二维码
	 * @param goodsInfoId
	 * @return
	 * @throws Exception
	 */
	@ResponseBody
	@RequestMapping(value = "cleanQr")
	public AjaxJson cleanQr(String goodsInfoId) throws Exception{
		AjaxJson j = new AjaxJson();
		//新增或编辑表单保存
		goodsService.cleanQr(goodsInfoId);//保存
		j.setSuccess(true);
		j.setMsg("保存货品模块成功");
		return j;
	}


	/**
	 * 根据批次号查询货品明细信息
	 * @param goods
	 * @return
	 */
    @ResponseBody
    @RequestMapping(value = "getGoodsInfoByGoods")
    public AjaxJson getGoodsInfoByGoods(Goods goods) {
        AjaxJson j = new AjaxJson();
        if (StringUtils.isNotBlank(goods.getBatchNum())) {
            Goods resultGoods = goodsService.findGoodsInfoByGoods(goods);
            j.put("goods", resultGoods);
            j.setSuccess(true);
            j.setMsg("查询成功");
        } else {
            j.setSuccess(false);
            j.setMsg("查询条件不能为空");
        }
        return j;
    }

	/**
	 * 根据货品明细id和二维码id查询货品信息和二维码信息
	 * @param goodsInfoId
	 * @param qrCodeId
	 * @return
	 */
	@ResponseBody
	@RequestMapping(value = "getGIAndQCByTwoId")
	public AjaxJson getGIAndQCByTwoId(String goodsInfoId, String qrCodeId) {
		AjaxJson j = new AjaxJson();
		//新增或编辑表单保存
		QrCode qrCode = goodsService.getGIAndQCByTwoId(goodsInfoId, qrCodeId);
		j.put("qrCode", qrCode);
		j.setSuccess(true);
		j.setMsg("查询成功成功");
		return j;
	}

}