package cn.gintone.controller;

import cn.gintone.controller.vo.PtUserInfoPageReqVO;
import cn.gintone.controller.vo.PtUserInfoRespVO;
import cn.gintone.controller.vo.PtUserInfoSaveReqVO;
import cn.gintone.entity.PtUserInfoDO;
import cn.gintone.service.PtUserInfoService;
import org.springframework.web.bind.annotation.*;
import javax.annotation.Resource;
import org.springframework.validation.annotation.Validated;
import org.springframework.security.access.prepost.PreAuthorize;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.Operation;

import javax.validation.constraints.*;
import javax.validation.*;
import javax.servlet.http.*;
import java.util.*;
import java.io.IOException;

import cn.iocoder.yudao.framework.common.pojo.PageParam;
import cn.iocoder.yudao.framework.common.pojo.PageResult;
import cn.iocoder.yudao.framework.common.pojo.CommonResult;
import cn.iocoder.yudao.framework.common.util.object.BeanUtils;
import static cn.iocoder.yudao.framework.common.pojo.CommonResult.success;

import cn.iocoder.yudao.framework.excel.core.util.ExcelUtils;

import cn.iocoder.yudao.framework.apilog.core.annotation.ApiAccessLog;
import static cn.iocoder.yudao.framework.apilog.core.enums.OperateTypeEnum.*;


@Tag(name = "管理后台 - 平台用户")
@RestController
@RequestMapping("/admin-api/ptinfo/pt-user-info")
@Validated
public class PtUserInfoController {

    @Resource
    private PtUserInfoService ptUserInfoService;

    @PostMapping("/create")
    @Operation(summary = "创建平台用户")
    @PreAuthorize("@ss.hasPermission('ptinfo:pt-user-info:create')")
    public CommonResult<Long> createPtUserInfo(@Valid @RequestBody PtUserInfoSaveReqVO createReqVO) {
        return success(ptUserInfoService.createPtUserInfo(createReqVO));
    }

    @PutMapping("/update")
    @Operation(summary = "更新平台用户")
    @PreAuthorize("@ss.hasPermission('ptinfo:pt-user-info:update')")
    public CommonResult<Boolean> updatePtUserInfo(@Valid @RequestBody PtUserInfoSaveReqVO updateReqVO) {
        ptUserInfoService.updatePtUserInfo(updateReqVO);
        return success(true);
    }

    @DeleteMapping("/delete")
    @Operation(summary = "删除平台用户")
    @Parameter(name = "id", description = "编号", required = true)
    @PreAuthorize("@ss.hasPermission('ptinfo:pt-user-info:delete')")
    public CommonResult<Boolean> deletePtUserInfo(@RequestParam("id") Long id) {
        ptUserInfoService.deletePtUserInfo(id);
        return success(true);
    }

    @GetMapping("/get")
    @Operation(summary = "获得平台用户")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    @PreAuthorize("@ss.hasPermission('ptinfo:pt-user-info:query')")
    public CommonResult<PtUserInfoRespVO> getPtUserInfo(@RequestParam("id") Long id) {
        PtUserInfoDO ptUserInfo = ptUserInfoService.getPtUserInfo(id);
        return success(BeanUtils.toBean(ptUserInfo, PtUserInfoRespVO.class));
    }

    @GetMapping("/page")
    @Operation(summary = "获得平台用户分页")
    @PreAuthorize("@ss.hasPermission('ptinfo:pt-user-info:query')")
    public CommonResult<PageResult<PtUserInfoRespVO>> getPtUserInfoPage(@Valid PtUserInfoPageReqVO pageReqVO) {
        PageResult<PtUserInfoDO> pageResult = ptUserInfoService.getPtUserInfoPage(pageReqVO);
        return success(BeanUtils.toBean(pageResult, PtUserInfoRespVO.class));
    }

    @GetMapping("/export-excel")
    @Operation(summary = "导出平台用户 Excel")
    @PreAuthorize("@ss.hasPermission('ptinfo:pt-user-info:export')")
    @ApiAccessLog(operateType = EXPORT)
    public void exportPtUserInfoExcel(@Valid PtUserInfoPageReqVO pageReqVO,
              HttpServletResponse response) throws IOException {
        pageReqVO.setPageSize(PageParam.PAGE_SIZE_NONE);
        List<PtUserInfoDO> list = ptUserInfoService.getPtUserInfoPage(pageReqVO).getList();
        // 导出 Excel
        ExcelUtils.write(response, "平台用户.xls", "数据", PtUserInfoRespVO.class,
                        BeanUtils.toBean(list, PtUserInfoRespVO.class));
    }


    @GetMapping("/syncPtUser")
    @Operation(summary = "同步平台部门信息")
    public CommonResult<String> syncPtUser() {
        return success(ptUserInfoService.syncPtUser());
    }
}