package cn.gintone.controller;

import cn.gintone.config.IotDbConfig;
import cn.gintone.controller.vo.KeyCodePageReqVO;
import cn.gintone.controller.vo.KeyCodeRespVO;
import cn.gintone.controller.vo.KeyCodeSaveReqVO;
import cn.gintone.dto.EncInfo;
import cn.gintone.dto.FileDecLogInfo;
import cn.gintone.dto.SpePeoLogInfo;
import cn.gintone.entity.KeyCodeDO;
import cn.gintone.iotdbUtils.FileIotDbUtil;
import cn.gintone.iotdbUtils.SpecialPeopleIotDbUtils;
import cn.gintone.service.KeyCodeService;
import cn.iocoder.yudao.framework.common.pojo.PageResult;
import cn.iocoder.yudao.module.system.service.auth.AdminAuthService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import javax.annotation.Resource;
import org.springframework.validation.annotation.Validated;
import org.springframework.security.access.prepost.PreAuthorize;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.Operation;

import javax.annotation.security.PermitAll;
import javax.validation.*;
import javax.servlet.http.*;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import cn.iocoder.yudao.framework.common.pojo.PageParam;
import cn.iocoder.yudao.framework.common.pojo.CommonResult;
import cn.iocoder.yudao.framework.common.util.object.BeanUtils;

import static cn.iocoder.yudao.framework.common.pojo.CommonResult.error;
import static cn.iocoder.yudao.framework.common.pojo.CommonResult.success;

import cn.iocoder.yudao.framework.excel.core.util.ExcelUtils;

import cn.iocoder.yudao.framework.apilog.core.annotation.ApiAccessLog;
import static cn.iocoder.yudao.framework.apilog.core.enums.OperateTypeEnum.*;


@Tag(name = "管理后台 - 公钥私钥管理")
@RestController
@RequestMapping("/admin-api/gintone/key-code")
@Validated
public class KeyCodeController {

    @Resource
    private KeyCodeService keyCodeService;
    @Autowired
    private IotDbConfig iotDbConfig;

    @Resource
    private AdminAuthService authService;

    @PostMapping("/create")
    @Operation(summary = "创建公钥私钥管理")
    @PreAuthorize("@ss.hasPermission('gintone:key-code:create')")
    public CommonResult<Long> createKeyCode(@Valid @RequestBody KeyCodeSaveReqVO createReqVO) {
        return success(keyCodeService.createKeyCode(createReqVO));
    }

    @PutMapping("/update")
    @Operation(summary = "更新公钥私钥管理")
    @PreAuthorize("@ss.hasPermission('gintone:key-code:update')")
    public CommonResult<Boolean> updateKeyCode(@Valid @RequestBody KeyCodeSaveReqVO updateReqVO) {
        keyCodeService.updateKeyCode(updateReqVO);
        return success(true);
    }

    @DeleteMapping("/delete")
    @Operation(summary = "删除公钥私钥管理")
    @Parameter(name = "id", description = "编号", required = true)
    @PreAuthorize("@ss.hasPermission('gintone:key-code:delete')")
    public CommonResult<Boolean> deleteKeyCode(@RequestParam("id") Long id) {
        keyCodeService.deleteKeyCode(id);
        return success(true);
    }

    @GetMapping("/get")
    @Operation(summary = "获得公钥私钥管理")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    @PreAuthorize("@ss.hasPermission('gintone:key-code:query')")
    public CommonResult<KeyCodeRespVO> getKeyCode(@RequestParam("id") Long id) {
        KeyCodeDO keyCode = keyCodeService.getKeyCode(id);
        return success(BeanUtils.toBean(keyCode, KeyCodeRespVO.class));
    }

    @GetMapping("/page")
    @Operation(summary = "获得公钥私钥管理分页")
    @PreAuthorize("@ss.hasPermission('gintone:key-code:query')")
    public CommonResult<PageResult<KeyCodeRespVO>> getKeyCodePage(@Valid KeyCodePageReqVO pageReqVO) {
        PageResult<KeyCodeDO> pageResult = keyCodeService.getKeyCodePage(pageReqVO);
        return success(BeanUtils.toBean(pageResult, KeyCodeRespVO.class));
    }

    @GetMapping("/export-excel")
    @Operation(summary = "导出公钥私钥管理 Excel")
    @PreAuthorize("@ss.hasPermission('gintone:key-code:export')")
    @ApiAccessLog(operateType = EXPORT)
    public void exportKeyCodeExcel(@Valid KeyCodePageReqVO pageReqVO,
              HttpServletResponse response) throws IOException {
        pageReqVO.setPageSize(PageParam.PAGE_SIZE_NONE);
        List<KeyCodeDO> list = keyCodeService.getKeyCodePage(pageReqVO).getList();
        // 导出 Excel
        ExcelUtils.write(response, "公钥私钥管理.xls", "数据", KeyCodeRespVO.class,
                        BeanUtils.toBean(list, KeyCodeRespVO.class));
    }

    @PostMapping("/initKey")
    @Operation(summary = "初始化公钥私钥")
    public CommonResult<String> initKey() {
        return success(keyCodeService.initKey());
    }

    @PermitAll
    @PostMapping("/rasEncryption")
    @Operation(summary = "外部RAS加密")
    public CommonResult<EncInfo> rasEncryption(@RequestBody Map<String, Object> requestMap) {
        EncInfo encInfo = keyCodeService.rasEncryption(requestMap);
        return success(encInfo);
    }

    @PermitAll
    @PostMapping("/rasDecrypt")
    @Operation(summary = "外部RAS解密")
    public CommonResult<Map<String, Object>> rasDecrypt(@RequestBody EncInfo encInfo, HttpServletRequest request) {
        String pdToken = request.getHeader("pdToken");
        String ip = request.getHeader("X-Forwarded-For");
        if (isInvalidIp(ip)) {
            ip = request.getHeader("Proxy-Client-IP");
        }
        if (isInvalidIp(ip)) {
            ip = request.getHeader("WL-Proxy-Client-IP");
        }
        if (isInvalidIp(ip)) {
            ip = request.getHeader("HTTP_CLIENT_IP");
        }
        if (isInvalidIp(ip)) {
            ip = request.getHeader("HTTP_X_FORWARDED_FOR");
        }
        if (isInvalidIp(ip)) {
            ip = request.getRemoteAddr();
        }
        Map<String, Object> map = keyCodeService.rasDecrypt(encInfo, ip, pdToken, true);
        return success(map);
    }
    @PostMapping("/smTwoEncryption")
    @Operation(summary = "sm2加密")
    public CommonResult<EncInfo> smTwoEncryption(@RequestBody Map<String, Object> requestMap) {
        EncInfo encInfo = keyCodeService.smTwoEncryption(requestMap);
        return success(encInfo);
    }

    @PostMapping("/smTwoDecrypt")
    @Operation(summary = "sm2解密")
    public CommonResult<Map<String, Object>> smTwoDecrypt(@RequestBody EncInfo encInfo, HttpServletRequest request) {
        String pdToken = request.getHeader("pdToken");
        String ip = request.getHeader("X-Forwarded-For");
        if (isInvalidIp(ip)) {
            ip = request.getHeader("Proxy-Client-IP");
        }
        if (isInvalidIp(ip)) {
            ip = request.getHeader("WL-Proxy-Client-IP");
        }
        if (isInvalidIp(ip)) {
            ip = request.getHeader("HTTP_CLIENT_IP");
        }
        if (isInvalidIp(ip)) {
            ip = request.getHeader("HTTP_X_FORWARDED_FOR");
        }
        if (isInvalidIp(ip)) {
            ip = request.getRemoteAddr();
        }

        // 处理多级代理情况（取第一个非 unknown 的 IP）
        if (ip != null && ip.contains(",")) {
            ip = Arrays.stream(ip.split(","))
                    .map(String::trim)
                    .filter(part -> !"unknown".equalsIgnoreCase(part))
                    .findFirst()
                    .orElse(request.getRemoteAddr());
        }

        Map<String, Object> map = keyCodeService.smTwoDecrypt(encInfo, ip, pdToken);
        return success(map);
    }

    private boolean isInvalidIp(String ip) {
        return ip == null || ip.isEmpty() || "unknown".equalsIgnoreCase(ip);
    }

    @GetMapping("/countSpeLogInfo")
    @Operation(summary = "重点人群申请解密日志条数")
    public CommonResult<Long> countSpeLogInfo(SpePeoLogInfo spePeoLogInfo) {
        long pageCount = SpecialPeopleIotDbUtils.countSpeLogInfo(iotDbConfig, spePeoLogInfo);
        return CommonResult.success(pageCount);
    }

    @GetMapping("/speLogInfoList")
    @Operation(summary = "重点人群申请日志记录")
    public CommonResult<List<SpePeoLogInfo>> speLogInfoList(SpePeoLogInfo spePeoLogInfo) {
        List<SpePeoLogInfo> spePeoLogInfoList = SpecialPeopleIotDbUtils.selectSpeLogInfo(iotDbConfig, spePeoLogInfo);
        return CommonResult.success(spePeoLogInfoList);
    }
}