package cn.gintone.controller;

import cn.gintone.controller.vo.DeviceTypePageReqVO;
import cn.gintone.controller.vo.DeviceTypeRespVO;
import cn.gintone.controller.vo.DeviceTypeSaveReqVO;
import cn.gintone.entity.DeviceTypeDO;
import cn.gintone.service.DeviceTypeService;
import org.springframework.web.bind.annotation.*;
import javax.annotation.Resource;
import org.springframework.validation.annotation.Validated;
import org.springframework.security.access.prepost.PreAuthorize;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.Operation;

import javax.validation.constraints.*;
import javax.validation.*;
import javax.servlet.http.*;
import java.util.*;
import java.io.IOException;

import cn.iocoder.yudao.framework.common.pojo.PageParam;
import cn.iocoder.yudao.framework.common.pojo.PageResult;
import cn.iocoder.yudao.framework.common.pojo.CommonResult;
import cn.iocoder.yudao.framework.common.util.object.BeanUtils;
import static cn.iocoder.yudao.framework.common.pojo.CommonResult.success;

import cn.iocoder.yudao.framework.excel.core.util.ExcelUtils;

import cn.iocoder.yudao.framework.apilog.core.annotation.ApiAccessLog;
import static cn.iocoder.yudao.framework.apilog.core.enums.OperateTypeEnum.*;


@Tag(name = "管理后台 - 设备类型")
@RestController
@RequestMapping("/admin-api/gintone/device-type")
@Validated
public class DeviceTypeController {

    @Resource
    private DeviceTypeService deviceTypeService;

    @PostMapping("/create")
    @Operation(summary = "创建设备类型")
    @PreAuthorize("@ss.hasPermission('gintone:device-type:create')")
    public CommonResult<Long> createDeviceType(@Valid @RequestBody DeviceTypeSaveReqVO createReqVO) {
        return success(deviceTypeService.createDeviceType(createReqVO));
    }

    @PutMapping("/update")
    @Operation(summary = "更新设备类型")
    @PreAuthorize("@ss.hasPermission('gintone:device-type:update')")
    public CommonResult<Boolean> updateDeviceType(@Valid @RequestBody DeviceTypeSaveReqVO updateReqVO) {
        deviceTypeService.updateDeviceType(updateReqVO);
        return success(true);
    }

    @DeleteMapping("/delete")
    @Operation(summary = "删除设备类型")
    @Parameter(name = "id", description = "编号", required = true)
    @PreAuthorize("@ss.hasPermission('gintone:device-type:delete')")
    public CommonResult<Boolean> deleteDeviceType(@RequestParam("id") Long id) {
        deviceTypeService.deleteDeviceType(id);
        return success(true);
    }

    @GetMapping("/get")
    @Operation(summary = "获得设备类型")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    @PreAuthorize("@ss.hasPermission('gintone:device-type:query')")
    public CommonResult<DeviceTypeRespVO> getDeviceType(@RequestParam("id") Long id) {
        DeviceTypeDO deviceType = deviceTypeService.getDeviceType(id);
        return success(BeanUtils.toBean(deviceType, DeviceTypeRespVO.class));
    }

    @GetMapping("/page")
    @Operation(summary = "获得设备类型分页")
    @PreAuthorize("@ss.hasPermission('gintone:device-type:query')")
    public CommonResult<PageResult<DeviceTypeRespVO>> getDeviceTypePage(@Valid DeviceTypePageReqVO pageReqVO) {
        PageResult<DeviceTypeDO> pageResult = deviceTypeService.getDeviceTypePage(pageReqVO);
        return success(BeanUtils.toBean(pageResult, DeviceTypeRespVO.class));
    }

    @GetMapping("/export-excel")
    @Operation(summary = "导出设备类型 Excel")
    @PreAuthorize("@ss.hasPermission('gintone:device-type:export')")
    @ApiAccessLog(operateType = EXPORT)
    public void exportDeviceTypeExcel(@Valid DeviceTypePageReqVO pageReqVO,
              HttpServletResponse response) throws IOException {
        pageReqVO.setPageSize(PageParam.PAGE_SIZE_NONE);
        List<DeviceTypeDO> list = deviceTypeService.getDeviceTypePage(pageReqVO).getList();
        // 导出 Excel
        ExcelUtils.write(response, "设备类型.xls", "数据", DeviceTypeRespVO.class,
                        BeanUtils.toBean(list, DeviceTypeRespVO.class));
    }

}