package cn.gintone.controller;

import cn.gintone.controller.vo.DeviceConnetRulePageReqVO;
import cn.gintone.controller.vo.DeviceConnetRuleRespVO;
import cn.gintone.controller.vo.DeviceConnetRuleSaveReqVO;
import cn.gintone.entity.DeviceConnetRuleDO;
import cn.gintone.service.DeviceConnetRuleService;
import cn.iocoder.yudao.framework.common.pojo.PageResult;
import org.springframework.web.bind.annotation.*;
import javax.annotation.Resource;
import org.springframework.validation.annotation.Validated;
import org.springframework.security.access.prepost.PreAuthorize;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.Operation;

import javax.validation.*;
import javax.servlet.http.*;
import java.io.IOException;
import java.util.List;

import cn.iocoder.yudao.framework.common.pojo.PageParam;
import cn.iocoder.yudao.framework.common.pojo.CommonResult;
import cn.iocoder.yudao.framework.common.util.object.BeanUtils;
import static cn.iocoder.yudao.framework.common.pojo.CommonResult.success;

import cn.iocoder.yudao.framework.excel.core.util.ExcelUtils;

import cn.iocoder.yudao.framework.apilog.core.annotation.ApiAccessLog;
import static cn.iocoder.yudao.framework.apilog.core.enums.OperateTypeEnum.*;

@Tag(name = "管理后台 - 设备连接规则")
@RestController
@RequestMapping("/admin-api/device/connet-rule")
@Validated
public class DeviceConnetRuleController {

    @Resource
    private DeviceConnetRuleService connetRuleService;

    @PostMapping("/create")
    @Operation(summary = "创建设备连接规则")
    @PreAuthorize("@ss.hasPermission('device:connet-rule:create')")
    public CommonResult<Long> createConnetRule(@Valid @RequestBody DeviceConnetRuleSaveReqVO createReqVO) {
        return success(connetRuleService.createConnetRule(createReqVO));
    }

    @PutMapping("/update")
    @Operation(summary = "更新设备连接规则")
    @PreAuthorize("@ss.hasPermission('device:connet-rule:update')")
    public CommonResult<Boolean> updateConnetRule(@Valid @RequestBody DeviceConnetRuleSaveReqVO updateReqVO) {
        connetRuleService.updateConnetRule(updateReqVO);
        return success(true);
    }

    @DeleteMapping("/delete")
    @Operation(summary = "删除设备连接规则")
    @Parameter(name = "id", description = "编号", required = true)
    @PreAuthorize("@ss.hasPermission('device:connet-rule:delete')")
    public CommonResult<Boolean> deleteConnetRule(@RequestParam("id") Long id) {
        connetRuleService.deleteConnetRule(id);
        return success(true);
    }

    @GetMapping("/get")
    @Operation(summary = "获得设备连接规则")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    @PreAuthorize("@ss.hasPermission('device:connet-rule:query')")
    public CommonResult<DeviceConnetRuleRespVO> getConnetRule(@RequestParam("id") Long id) {
        DeviceConnetRuleDO connetRule = connetRuleService.getConnetRule(id);
        return success(BeanUtils.toBean(connetRule, DeviceConnetRuleRespVO.class));
    }

    @GetMapping("/page")
    @Operation(summary = "获得设备连接规则分页")
    @PreAuthorize("@ss.hasPermission('device:connet-rule:query')")
    public CommonResult<PageResult<DeviceConnetRuleRespVO>> getConnetRulePage(@Valid DeviceConnetRulePageReqVO pageReqVO) {
        PageResult<DeviceConnetRuleDO> pageResult = connetRuleService.getConnetRulePage(pageReqVO);
        return success(BeanUtils.toBean(pageResult, DeviceConnetRuleRespVO.class));
    }

    @GetMapping("/export-excel")
    @Operation(summary = "导出设备连接规则 Excel")
    @PreAuthorize("@ss.hasPermission('device:connet-rule:export')")
    @ApiAccessLog(operateType = EXPORT)
    public void exportConnetRuleExcel(@Valid DeviceConnetRulePageReqVO pageReqVO,
              HttpServletResponse response) throws IOException {
        pageReqVO.setPageSize(PageParam.PAGE_SIZE_NONE);
        List<DeviceConnetRuleDO> list = connetRuleService.getConnetRulePage(pageReqVO).getList();
        // 导出 Excel
        ExcelUtils.write(response, "设备连接规则.xls", "数据", DeviceConnetRuleRespVO.class,
                        BeanUtils.toBean(list, DeviceConnetRuleRespVO.class));
    }

}