package cn.gintone.controller;

import cn.gintone.controller.vo.DeviceReqConfListReqVO;
import cn.gintone.controller.vo.DeviceReqConfRespVO;
import cn.gintone.controller.vo.DeviceReqConfSaveReqVO;
import cn.gintone.entity.DeviceReqConfDO;
import cn.gintone.service.DeviceReqConfService;
import cn.iocoder.yudao.module.system.controller.admin.myUtils.MyHttpUtils;
import org.springframework.web.bind.annotation.*;
import javax.annotation.Resource;
import org.springframework.validation.annotation.Validated;
import org.springframework.security.access.prepost.PreAuthorize;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.Operation;

import javax.annotation.security.PermitAll;
import javax.validation.constraints.*;
import javax.validation.*;
import javax.servlet.http.*;
import java.util.*;
import java.io.IOException;

import cn.iocoder.yudao.framework.common.pojo.PageParam;
import cn.iocoder.yudao.framework.common.pojo.PageResult;
import cn.iocoder.yudao.framework.common.pojo.CommonResult;
import cn.iocoder.yudao.framework.common.util.object.BeanUtils;
import static cn.iocoder.yudao.framework.common.pojo.CommonResult.success;

import cn.iocoder.yudao.framework.excel.core.util.ExcelUtils;

import cn.iocoder.yudao.framework.apilog.core.annotation.ApiAccessLog;
import static cn.iocoder.yudao.framework.apilog.core.enums.OperateTypeEnum.*;


@Tag(name = "管理后台 - 物联网设备请求配置")
@RestController
@RequestMapping("/admin-api/sec/device-req-conf")
@Validated
public class DeviceReqConfController {

    @Resource
    private DeviceReqConfService deviceReqConfService;

    @PostMapping("/create")
    @Operation(summary = "创建物联网设备请求配置")
    @PreAuthorize("@ss.hasPermission('sec:device-req-conf:create')")
    public CommonResult<Long> createDeviceReqConf(@Valid @RequestBody DeviceReqConfSaveReqVO createReqVO) {
        return success(deviceReqConfService.createDeviceReqConf(createReqVO));
    }

    @PutMapping("/update")
    @Operation(summary = "更新物联网设备请求配置")
    @PreAuthorize("@ss.hasPermission('sec:device-req-conf:update')")
    public CommonResult<Boolean> updateDeviceReqConf(@Valid @RequestBody DeviceReqConfSaveReqVO updateReqVO) {
        deviceReqConfService.updateDeviceReqConf(updateReqVO);
        return success(true);
    }

    @DeleteMapping("/delete")
    @Operation(summary = "删除物联网设备请求配置")
    @Parameter(name = "id", description = "编号", required = true)
    @PreAuthorize("@ss.hasPermission('sec:device-req-conf:delete')")
    public CommonResult<Boolean> deleteDeviceReqConf(@RequestParam("id") Long id) {
        deviceReqConfService.deleteDeviceReqConf(id);
        return success(true);
    }

    @GetMapping("/get")
    @Operation(summary = "获得物联网设备请求配置")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    @PreAuthorize("@ss.hasPermission('sec:device-req-conf:query')")
    public CommonResult<DeviceReqConfRespVO> getDeviceReqConf(@RequestParam("id") Long id) {
        DeviceReqConfDO deviceReqConf = deviceReqConfService.getDeviceReqConf(id);
        return success(BeanUtils.toBean(deviceReqConf, DeviceReqConfRespVO.class));
    }

    @GetMapping("/list")
    @Operation(summary = "获得物联网设备请求配置列表")
    @PreAuthorize("@ss.hasPermission('sec:device-req-conf:query')")
    public CommonResult<List<DeviceReqConfRespVO>> getDeviceReqConfList(@Valid DeviceReqConfListReqVO listReqVO) {
        List<DeviceReqConfDO> list = deviceReqConfService.getDeviceReqConfList(listReqVO);
        return success(BeanUtils.toBean(list, DeviceReqConfRespVO.class));
    }

    @GetMapping("/export-excel")
    @Operation(summary = "导出物联网设备请求配置 Excel")
    @PreAuthorize("@ss.hasPermission('sec:device-req-conf:export')")
    @ApiAccessLog(operateType = EXPORT)
    public void exportDeviceReqConfExcel(@Valid DeviceReqConfListReqVO listReqVO,
              HttpServletResponse response) throws IOException {
        List<DeviceReqConfDO> list = deviceReqConfService.getDeviceReqConfList(listReqVO);
        // 导出 Excel
        ExcelUtils.write(response, "物联网设备请求配置.xls", "数据", DeviceReqConfRespVO.class,
                        BeanUtils.toBean(list, DeviceReqConfRespVO.class));
    }


    @PermitAll
    @GetMapping("/syncPtUser")
    @Operation(summary = "同步平台设备信息")
    public CommonResult<String> syncPtDevice() {
        deviceReqConfService.exportPtDeviceType(); // 同步设备类型
        deviceReqConfService.exportPtDevice(); // 同步设备
        return success("同步成功");
    }
}