package cn.gintone.controller;

import cn.gintone.controller.vo.DesCorporationPageReqVO;
import cn.gintone.controller.vo.DesCorporationRespVO;
import cn.gintone.controller.vo.DesCorporationSaveReqVO;
import cn.gintone.dto.DesInfo;
import cn.gintone.entity.DesCorporationDO;
import cn.gintone.service.DesCorporationService;
import cn.iocoder.yudao.framework.common.pojo.PageResult;
import org.springframework.web.bind.annotation.*;
import javax.annotation.Resource;
import org.springframework.validation.annotation.Validated;
import org.springframework.security.access.prepost.PreAuthorize;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.Operation;

import javax.annotation.security.PermitAll;
import javax.validation.*;
import javax.servlet.http.*;
import java.io.IOException;
import java.util.List;
import java.util.Map;

import cn.iocoder.yudao.framework.common.pojo.PageParam;
import cn.iocoder.yudao.framework.common.pojo.CommonResult;
import cn.iocoder.yudao.framework.common.util.object.BeanUtils;
import static cn.iocoder.yudao.framework.common.pojo.CommonResult.success;

import cn.iocoder.yudao.framework.excel.core.util.ExcelUtils;

import cn.iocoder.yudao.framework.apilog.core.annotation.ApiAccessLog;
import static cn.iocoder.yudao.framework.apilog.core.enums.OperateTypeEnum.*;


@Tag(name = "管理后台 - 法人脱敏")
@RestController
@RequestMapping("/admin-api/sec/des-corporation")
@Validated
public class DesCorporationController {

    @Resource
    private DesCorporationService desCorporationService;

    @PostMapping("/create")
    @Operation(summary = "创建法人脱敏")
    @PreAuthorize("@ss.hasPermission('sec:des-corporation:create')")
    public CommonResult<Long> createDesCorporation(@Valid @RequestBody DesCorporationSaveReqVO createReqVO) {
        return success(desCorporationService.createDesCorporation(createReqVO));
    }

    @PutMapping("/update")
    @Operation(summary = "更新法人脱敏")
    @PreAuthorize("@ss.hasPermission('sec:des-corporation:update')")
    public CommonResult<Boolean> updateDesCorporation(@Valid @RequestBody DesCorporationSaveReqVO updateReqVO) {
        desCorporationService.updateDesCorporation(updateReqVO);
        return success(true);
    }

    @DeleteMapping("/delete")
    @Operation(summary = "删除法人脱敏")
    @Parameter(name = "id", description = "编号", required = true)
    @PreAuthorize("@ss.hasPermission('sec:des-corporation:delete')")
    public CommonResult<Boolean> deleteDesCorporation(@RequestParam("id") Long id) {
        desCorporationService.deleteDesCorporation(id);
        return success(true);
    }

    @GetMapping("/get")
    @Operation(summary = "获得法人脱敏")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    @PreAuthorize("@ss.hasPermission('sec:des-corporation:query')")
    public CommonResult<DesCorporationRespVO> getDesCorporation(@RequestParam("id") Long id) {
        DesCorporationDO desCorporation = desCorporationService.getDesCorporation(id);
        return success(BeanUtils.toBean(desCorporation, DesCorporationRespVO.class));
    }

    @GetMapping("/page")
    @Operation(summary = "获得法人脱敏分页")
    @PreAuthorize("@ss.hasPermission('sec:des-corporation:query')")
    public CommonResult<PageResult<DesCorporationRespVO>> getDesCorporationPage(@Valid DesCorporationPageReqVO pageReqVO) {
        PageResult<DesCorporationDO> pageResult = desCorporationService.getDesCorporationPage(pageReqVO);
        return success(BeanUtils.toBean(pageResult, DesCorporationRespVO.class));
    }

    @GetMapping("/export-excel")
    @Operation(summary = "导出法人脱敏 Excel")
    @PreAuthorize("@ss.hasPermission('sec:des-corporation:export')")
    @ApiAccessLog(operateType = EXPORT)
    public void exportDesCorporationExcel(@Valid DesCorporationPageReqVO pageReqVO,
              HttpServletResponse response) throws IOException {
        pageReqVO.setPageSize(PageParam.PAGE_SIZE_NONE);
        List<DesCorporationDO> list = desCorporationService.getDesCorporationPage(pageReqVO).getList();
        // 导出 Excel
        ExcelUtils.write(response, "法人脱敏.xls", "数据", DesCorporationRespVO.class,
                        BeanUtils.toBean(list, DesCorporationRespVO.class));
    }

    @GetMapping("/checkName")
    @Operation(summary = "验证属性是否存在")
    public CommonResult<Boolean> checkName(@RequestParam("name") String name, @RequestParam(value = "id", required = false) Long id) {
        return success(desCorporationService.checkName(name, id));
    }


    @PostMapping("/corporationRuleDes")
    @Operation(summary = "法人脱敏")
    public CommonResult<String> corporationRuleDes(@RequestBody DesInfo desInfo) {
        String str = desCorporationService.corporationRuleDes(desInfo.getJsonStr());
        return CommonResult.success(str);
    }

    @PermitAll
    @PostMapping("/corporationRuleDesMap")
    @Operation(summary = "法人脱敏")
    public CommonResult<Map<String, Object>> corporationRuleDesMap(@RequestBody Map<String, Object> map) {
        Map<String, Object> resultMap = desCorporationService.corporationRuleDesMap(map);
        return CommonResult.success(resultMap);
    }

    @PostMapping("/corporationRuleDesList")
    @Operation(summary = "法人批量脱敏")
    public CommonResult<String> corporationRuleDesList(@RequestBody DesInfo desInfo) {
        String str = desCorporationService.corporationRuleDesArr(desInfo.getJsonArrStr());
        return CommonResult.success(str);
    }


    @PermitAll
    @PostMapping("/corporationRuleDesListMap")
    @Operation(summary = "法人批量脱敏")
    public CommonResult<List<Map<String, Object>>> corporationRuleDesListMap(@RequestBody List<Map<String, Object>> lisetMap) {
        List<Map<String, Object>> resultList = desCorporationService.corporationRuleDesListMap(lisetMap);
        return CommonResult.success(resultList);
    }



}