package cn.gintone.controller;

import cn.gintone.controller.vo.UserDesRulePageReqVO;
import cn.gintone.controller.vo.UserDesRuleRespVO;
import cn.gintone.controller.vo.UserDesRuleSaveReqVO;
import cn.gintone.dto.DesInfo;
import cn.gintone.entity.UserDesRuleDO;
import cn.gintone.service.UserDesRuleService;
import org.springframework.web.bind.annotation.*;
import javax.annotation.Resource;
import org.springframework.validation.annotation.Validated;
import org.springframework.security.access.prepost.PreAuthorize;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.Operation;

import javax.validation.*;
import javax.servlet.http.*;
import java.io.IOException;
import java.util.List;
import java.util.Map;

import cn.iocoder.yudao.framework.common.pojo.PageParam;
import cn.iocoder.yudao.framework.common.pojo.PageResult;
import cn.iocoder.yudao.framework.common.pojo.CommonResult;
import cn.iocoder.yudao.framework.common.util.object.BeanUtils;
import static cn.iocoder.yudao.framework.common.pojo.CommonResult.success;

import cn.iocoder.yudao.framework.excel.core.util.ExcelUtils;

import cn.iocoder.yudao.framework.apilog.core.annotation.ApiAccessLog;
import static cn.iocoder.yudao.framework.apilog.core.enums.OperateTypeEnum.*;


@Tag(name = "管理后台 - 人员脱敏规则")
@RestController
@RequestMapping("/admin-api/sec/user-des-rule")
@Validated
public class UserDesRuleController {

    @Resource
    private UserDesRuleService userDesRuleService;

    @PostMapping("/create")
    @Operation(summary = "创建人员脱敏规则")
    @PreAuthorize("@ss.hasPermission('sec:user-des-rule:create')")
    public CommonResult<Long> createUserDesRule(@Valid @RequestBody UserDesRuleSaveReqVO createReqVO) {
        return success(userDesRuleService.createUserDesRule(createReqVO));
    }

    @PutMapping("/update")
    @Operation(summary = "更新人员脱敏规则")
    @PreAuthorize("@ss.hasPermission('sec:user-des-rule:update')")
    public CommonResult<Boolean> updateUserDesRule(@Valid @RequestBody UserDesRuleSaveReqVO updateReqVO) {
        userDesRuleService.updateUserDesRule(updateReqVO);
        return success(true);
    }

    @DeleteMapping("/delete")
    @Operation(summary = "删除人员脱敏规则")
    @Parameter(name = "id", description = "编号", required = true)
    @PreAuthorize("@ss.hasPermission('sec:user-des-rule:delete')")
    public CommonResult<Boolean> deleteUserDesRule(@RequestParam("id") Long id) {
        userDesRuleService.deleteUserDesRule(id);
        return success(true);
    }

    @GetMapping("/get")
    @Operation(summary = "获得人员脱敏规则")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    @PreAuthorize("@ss.hasPermission('sec:user-des-rule:query')")
    public CommonResult<UserDesRuleRespVO> getUserDesRule(@RequestParam("id") Long id) {
        UserDesRuleDO userDesRule = userDesRuleService.getUserDesRule(id);
        return success(BeanUtils.toBean(userDesRule, UserDesRuleRespVO.class));
    }

    @GetMapping("/page")
    @Operation(summary = "获得人员脱敏规则分页")
    @PreAuthorize("@ss.hasPermission('sec:user-des-rule:query')")
    public CommonResult<PageResult<UserDesRuleRespVO>> getUserDesRulePage(@Valid UserDesRulePageReqVO pageReqVO) {
        PageResult<UserDesRuleDO> pageResult = userDesRuleService.getUserDesRulePage(pageReqVO);
        return success(BeanUtils.toBean(pageResult, UserDesRuleRespVO.class));
    }

    @GetMapping("/export-excel")
    @Operation(summary = "导出人员脱敏规则 Excel")
    @PreAuthorize("@ss.hasPermission('sec:user-des-rule:export')")
    @ApiAccessLog(operateType = EXPORT)
    public void exportUserDesRuleExcel(@Valid UserDesRulePageReqVO pageReqVO,
              HttpServletResponse response) throws IOException {
        pageReqVO.setPageSize(PageParam.PAGE_SIZE_NONE);
        List<UserDesRuleDO> list = userDesRuleService.getUserDesRulePage(pageReqVO).getList();
        // 导出 Excel
        ExcelUtils.write(response, "人员脱敏规则.xls", "数据", UserDesRuleRespVO.class,
                        BeanUtils.toBean(list, UserDesRuleRespVO.class));
    }


    @GetMapping("/checkName")
    @Operation(summary = "验证属性是否存在")
    public CommonResult<Boolean> checkName(@RequestParam("name") String name, @RequestParam(value = "id", required = false) Long id) {
        return success(userDesRuleService.checkName(name, id));
    }


    @PostMapping("/userRuleDes")
    @Operation(summary = "人员脱敏")
    public CommonResult<String> userRuleDes(@RequestBody DesInfo desInfo) {
        String str = userDesRuleService.userRuleDes(desInfo.getJsonStr());
        return CommonResult.success(str);
    }

    @PostMapping("/userRuleDesMap")
    @Operation(summary = "人员脱敏")
    public CommonResult<Map<String, Object>> userRuleDesMap(@RequestBody Map<String, Object> map) {
        Map<String, Object> resultMap = userDesRuleService.userRuleDesMap(map);
        return CommonResult.success(resultMap);
    }

    @PostMapping("/userRuleDesList")
    @Operation(summary = "人员批量脱敏")
    public CommonResult<String> userRuleDesList(@RequestBody DesInfo desInfo) {
        String str = userDesRuleService.userRuleDesArr(desInfo.getJsonArrStr());
        return CommonResult.success(str);
    }


    @PostMapping("/userRuleDesListMap")
    @Operation(summary = "人员批量脱敏")
    public CommonResult<List<Map<String, Object>>> userRuleDesListMap(@RequestBody List<Map<String, Object>> lisetMap) {
        List<Map<String, Object>> resultList = userDesRuleService.userRuleDesListMap(lisetMap);
        return CommonResult.success(resultList);
    }
}