package cn.gintone.controller;

import cn.gintone.controller.vo.KeyCodePageReqVO;
import cn.gintone.controller.vo.KeyCodeRespVO;
import cn.gintone.controller.vo.KeyCodeSaveReqVO;
import cn.gintone.dto.EncInfo;
import cn.gintone.entity.KeyCodeDO;
import cn.gintone.service.KeyCodeService;
import cn.iocoder.yudao.framework.common.pojo.PageResult;
import org.springframework.web.bind.annotation.*;
import javax.annotation.Resource;
import org.springframework.validation.annotation.Validated;
import org.springframework.security.access.prepost.PreAuthorize;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.Operation;

import javax.validation.*;
import javax.servlet.http.*;
import java.io.IOException;
import java.util.List;

import cn.iocoder.yudao.framework.common.pojo.PageParam;
import cn.iocoder.yudao.framework.common.pojo.CommonResult;
import cn.iocoder.yudao.framework.common.util.object.BeanUtils;
import static cn.iocoder.yudao.framework.common.pojo.CommonResult.success;

import cn.iocoder.yudao.framework.excel.core.util.ExcelUtils;

import cn.iocoder.yudao.framework.apilog.core.annotation.ApiAccessLog;
import static cn.iocoder.yudao.framework.apilog.core.enums.OperateTypeEnum.*;


@Tag(name = "管理后台 - 公钥私钥管理")
@RestController
@RequestMapping("/admin-api/gintone/key-code")
@Validated
public class KeyCodeController {

    @Resource
    private KeyCodeService keyCodeService;

    @PostMapping("/create")
    @Operation(summary = "创建公钥私钥管理")
    @PreAuthorize("@ss.hasPermission('gintone:key-code:create')")
    public CommonResult<Long> createKeyCode(@Valid @RequestBody KeyCodeSaveReqVO createReqVO) {
        return success(keyCodeService.createKeyCode(createReqVO));
    }

    @PutMapping("/update")
    @Operation(summary = "更新公钥私钥管理")
    @PreAuthorize("@ss.hasPermission('gintone:key-code:update')")
    public CommonResult<Boolean> updateKeyCode(@Valid @RequestBody KeyCodeSaveReqVO updateReqVO) {
        keyCodeService.updateKeyCode(updateReqVO);
        return success(true);
    }

    @DeleteMapping("/delete")
    @Operation(summary = "删除公钥私钥管理")
    @Parameter(name = "id", description = "编号", required = true)
    @PreAuthorize("@ss.hasPermission('gintone:key-code:delete')")
    public CommonResult<Boolean> deleteKeyCode(@RequestParam("id") Long id) {
        keyCodeService.deleteKeyCode(id);
        return success(true);
    }

    @GetMapping("/get")
    @Operation(summary = "获得公钥私钥管理")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    @PreAuthorize("@ss.hasPermission('gintone:key-code:query')")
    public CommonResult<KeyCodeRespVO> getKeyCode(@RequestParam("id") Long id) {
        KeyCodeDO keyCode = keyCodeService.getKeyCode(id);
        return success(BeanUtils.toBean(keyCode, KeyCodeRespVO.class));
    }

    @GetMapping("/page")
    @Operation(summary = "获得公钥私钥管理分页")
    @PreAuthorize("@ss.hasPermission('gintone:key-code:query')")
    public CommonResult<PageResult<KeyCodeRespVO>> getKeyCodePage(@Valid KeyCodePageReqVO pageReqVO) {
        PageResult<KeyCodeDO> pageResult = keyCodeService.getKeyCodePage(pageReqVO);
        return success(BeanUtils.toBean(pageResult, KeyCodeRespVO.class));
    }

    @GetMapping("/export-excel")
    @Operation(summary = "导出公钥私钥管理 Excel")
    @PreAuthorize("@ss.hasPermission('gintone:key-code:export')")
    @ApiAccessLog(operateType = EXPORT)
    public void exportKeyCodeExcel(@Valid KeyCodePageReqVO pageReqVO,
              HttpServletResponse response) throws IOException {
        pageReqVO.setPageSize(PageParam.PAGE_SIZE_NONE);
        List<KeyCodeDO> list = keyCodeService.getKeyCodePage(pageReqVO).getList();
        // 导出 Excel
        ExcelUtils.write(response, "公钥私钥管理.xls", "数据", KeyCodeRespVO.class,
                        BeanUtils.toBean(list, KeyCodeRespVO.class));
    }

    @PostMapping("/initKey")
    @Operation(summary = "初始化公钥私钥")
    public CommonResult<String> initKey() {
        return success(keyCodeService.initKey());
    }

    @PostMapping("/rasEncryption")
    @Operation(summary = "RAS加密")
    public CommonResult<EncInfo> rasEncryption(String info) {
        EncInfo encInfo = keyCodeService.rasEncryption(info);
        return success(encInfo);
    }

    @PostMapping("/rasDecrypt")
    @Operation(summary = "RAS解密")
    public CommonResult<String> rasDecrypt(@RequestBody EncInfo encInfo) {
        String info = keyCodeService.rasDecrypt(encInfo);
        return success(info);
    }
}