package cn.gintone.controller;

import cn.gintone.controller.vo.VisitPageReqVO;
import cn.gintone.controller.vo.VisitRespVO;
import cn.gintone.controller.vo.VisitSaveReqVO;
import cn.gintone.entity.VisitDO;
import cn.gintone.service.VisitService;
import cn.iocoder.yudao.framework.common.pojo.PageResult;
import org.springframework.web.bind.annotation.*;
import javax.annotation.Resource;
import org.springframework.validation.annotation.Validated;
import org.springframework.security.access.prepost.PreAuthorize;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.Operation;

import javax.validation.*;
import javax.servlet.http.*;
import java.io.IOException;
import java.util.List;

import cn.iocoder.yudao.framework.common.pojo.PageParam;
import cn.iocoder.yudao.framework.common.pojo.CommonResult;
import cn.iocoder.yudao.framework.common.util.object.BeanUtils;
import static cn.iocoder.yudao.framework.common.pojo.CommonResult.success;

import cn.iocoder.yudao.framework.excel.core.util.ExcelUtils;

import cn.iocoder.yudao.framework.apilog.core.annotation.ApiAccessLog;
import static cn.iocoder.yudao.framework.apilog.core.enums.OperateTypeEnum.*;


@Tag(name = "管理后台 - 接口安全访问策略")
@RestController
@RequestMapping("/admin-api/visit/visit")
@Validated
public class VisitController {

    @Resource
    private VisitService visitService;

    @PostMapping("/create")
    @Operation(summary = "创建接口安全访问策略")
    @PreAuthorize("@ss.hasPermission('visit:visit:create')")
    public CommonResult<Long> createVisit(@Valid @RequestBody VisitSaveReqVO createReqVO) {
        return success(visitService.createVisit(createReqVO));
    }

    @PutMapping("/update")
    @Operation(summary = "更新接口安全访问策略")
    @PreAuthorize("@ss.hasPermission('visit:visit:update')")
    public CommonResult<Boolean> updateVisit(@Valid @RequestBody VisitSaveReqVO updateReqVO) {
        visitService.updateVisit(updateReqVO);
        return success(true);
    }

    @DeleteMapping("/delete")
    @Operation(summary = "删除接口安全访问策略")
    @Parameter(name = "id", description = "编号", required = true)
    @PreAuthorize("@ss.hasPermission('visit:visit:delete')")
    public CommonResult<Boolean> deleteVisit(@RequestParam("id") Long id) {
        visitService.deleteVisit(id);
        return success(true);
    }

    @GetMapping("/get")
    @Operation(summary = "获得接口安全访问策略")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    @PreAuthorize("@ss.hasPermission('visit:visit:query')")
    public CommonResult<VisitRespVO> getVisit(@RequestParam("id") Long id) {
        VisitDO visit = visitService.getVisit(id);
        return success(BeanUtils.toBean(visit, VisitRespVO.class));
    }

    @GetMapping("/page")
    @Operation(summary = "获得接口安全访问策略分页")
    @PreAuthorize("@ss.hasPermission('visit:visit:query')")
    public CommonResult<PageResult<VisitRespVO>> getVisitPage(@Valid VisitPageReqVO pageReqVO) {
        PageResult<VisitDO> pageResult = visitService.getVisitPage(pageReqVO);
        return success(BeanUtils.toBean(pageResult, VisitRespVO.class));
    }

    @GetMapping("/export-excel")
    @Operation(summary = "导出接口安全访问策略 Excel")
    @PreAuthorize("@ss.hasPermission('visit:visit:export')")
    @ApiAccessLog(operateType = EXPORT)
    public void exportVisitExcel(@Valid VisitPageReqVO pageReqVO,
              HttpServletResponse response) throws IOException {
        pageReqVO.setPageSize(PageParam.PAGE_SIZE_NONE);
        List<VisitDO> list = visitService.getVisitPage(pageReqVO).getList();
        // 导出 Excel
        ExcelUtils.write(response, "接口安全访问策略.xls", "数据", VisitRespVO.class,
                        BeanUtils.toBean(list, VisitRespVO.class));
    }

}