package cn.gintone.controller;

import cn.gintone.controller.vo.MyTablePageReqVO;
import cn.gintone.controller.vo.MyTableRespVO;
import cn.gintone.controller.vo.MyTableSaveReqVO;
import cn.gintone.entity.MyTableDO;
import cn.gintone.service.MyTableService;
import cn.iocoder.yudao.framework.common.pojo.PageResult;
import org.springframework.web.bind.annotation.*;
import javax.annotation.Resource;
import org.springframework.validation.annotation.Validated;
import org.springframework.security.access.prepost.PreAuthorize;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.Operation;

import javax.validation.*;
import javax.servlet.http.*;
import java.util.*;
import java.io.IOException;

import cn.iocoder.yudao.framework.common.pojo.PageParam;
import cn.iocoder.yudao.framework.common.pojo.CommonResult;
import cn.iocoder.yudao.framework.common.util.object.BeanUtils;
import static cn.iocoder.yudao.framework.common.pojo.CommonResult.success;

import cn.iocoder.yudao.framework.excel.core.util.ExcelUtils;

import cn.iocoder.yudao.framework.apilog.core.annotation.ApiAccessLog;
import static cn.iocoder.yudao.framework.apilog.core.enums.OperateTypeEnum.*;


@Tag(name = "管理后台 - 我的测试表")
@RestController
@RequestMapping("/admin-api/sec/my-table")
@Validated
public class MyTableController {

    @Resource
    private MyTableService myTableService;

    @PostMapping("/create")
    @Operation(summary = "创建我的测试表")
    @PreAuthorize("@ss.hasPermission('sec:my-table:create')")
    public CommonResult<Long> createMyTable(@Valid @RequestBody MyTableSaveReqVO createReqVO) {
        return success(myTableService.createMyTable(createReqVO));
    }

    @PutMapping("/update")
    @Operation(summary = "更新我的测试表")
    @PreAuthorize("@ss.hasPermission('sec:my-table:update')")
    public CommonResult<Boolean> updateMyTable(@Valid @RequestBody MyTableSaveReqVO updateReqVO) {
        myTableService.updateMyTable(updateReqVO);
        return success(true);
    }

    @DeleteMapping("/delete")
    @Operation(summary = "删除我的测试表")
    @Parameter(name = "id", description = "编号", required = true)
    @PreAuthorize("@ss.hasPermission('sec:my-table:delete')")
    public CommonResult<Boolean> deleteMyTable(@RequestParam("id") Long id) {
        myTableService.deleteMyTable(id);
        return success(true);
    }

    @GetMapping("/get")
    @Operation(summary = "获得我的测试表")
    @Parameter(name = "id", description = "编号", required = true, example = "1024")
    @PreAuthorize("@ss.hasPermission('sec:my-table:query')")
    public CommonResult<MyTableRespVO> getMyTable(@RequestParam("id") Long id) {
        MyTableDO myTable = myTableService.getMyTable(id);
        return success(BeanUtils.toBean(myTable, MyTableRespVO.class));
    }

    @GetMapping("/page")
    @Operation(summary = "获得我的测试表分页")
    @PreAuthorize("@ss.hasPermission('sec:my-table:query')")
    public CommonResult<PageResult<MyTableRespVO>> getMyTablePage(@Valid MyTablePageReqVO pageReqVO) {
        PageResult<MyTableDO> pageResult = myTableService.getMyTablePage(pageReqVO);
        return success(BeanUtils.toBean(pageResult, MyTableRespVO.class));
    }

    @GetMapping("/export-excel")
    @Operation(summary = "导出我的测试表 Excel")
    @PreAuthorize("@ss.hasPermission('sec:my-table:export')")
    @ApiAccessLog(operateType = EXPORT)
    public void exportMyTableExcel(@Valid MyTablePageReqVO pageReqVO,
              HttpServletResponse response) throws IOException {
        pageReqVO.setPageSize(PageParam.PAGE_SIZE_NONE);
        List<MyTableDO> list = myTableService.getMyTablePage(pageReqVO).getList();
        // 导出 Excel
        ExcelUtils.write(response, "我的测试表.xls", "数据", MyTableRespVO.class,
                        BeanUtils.toBean(list, MyTableRespVO.class));
    }

}