import 'dart:async';
import 'dart:convert';
import 'dart:typed_data';

import 'package:flutter/material.dart';
import 'package:flutter_easyrefresh/easy_refresh.dart';
import 'package:special_equipment_flutter/common/color_const.dart';

/// 冲上云霄Header
class TaurusHeaders extends Header {
  /// Key
  final Key? key;

  /// 背景色
  final Color? backgroundColor;

  final LinkHeaderNotifier linkNotifier = LinkHeaderNotifier();

  TaurusHeaders({
    this.key,
    this.backgroundColor = ColorConst.blue4Color,
    bool enableHapticFeedback = false,
    Duration completeDuration = const Duration(seconds: 1),
  }) : super(
          extent: 90.0,
          triggerDistance: 90.0,
          float: false,
          enableHapticFeedback: enableHapticFeedback,
          enableInfiniteRefresh: false,
          completeDuration: completeDuration,
        );

  @override
  Widget contentBuilder(
      BuildContext context,
      RefreshMode refreshState,
      double pulledExtent,
      double refreshTriggerPullDistance,
      double refreshIndicatorExtent,
      AxisDirection axisDirection,
      bool float,
      Duration? completeDuration,
      bool enableInfiniteRefresh,
      bool success,
      bool noMore) {
    // 不能为水平方向以及反向
    assert(axisDirection == AxisDirection.down,
        'Widget can only be vertical and cannot be reversed');
    linkNotifier.contentBuilder(
        context,
        refreshState,
        pulledExtent,
        refreshTriggerPullDistance,
        refreshIndicatorExtent,
        axisDirection,
        float,
        completeDuration,
        enableInfiniteRefresh,
        success,
        noMore);
    return TaurusHeaderWidget(
      key: key,
      backgroundColor: backgroundColor,
      linkNotifier: linkNotifier,
    );
  }
}

/// 冲上云霄组件
class TaurusHeaderWidget extends StatefulWidget {
  /// 背景色
  final Color? backgroundColor;

  final LinkHeaderNotifier linkNotifier;

  const TaurusHeaderWidget({
    Key? key,
    this.backgroundColor = Colors.blue,
    required this.linkNotifier,
  }) : super(key: key);

  @override
  TaurusHeaderWidgetState createState() {
    return TaurusHeaderWidgetState();
  }
}

class TaurusHeaderWidgetState extends State<TaurusHeaderWidget> {
  RefreshMode get _refreshState => widget.linkNotifier.refreshState;

  double get _pulledExtent => widget.linkNotifier.pulledExtent;

  double get _indicatorExtent => widget.linkNotifier.refreshIndicatorExtent;

  // 图片资源
  String _airplaneBase64 =
      "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";
  String _cloudsCenterBase64 =
      "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";
  String _cloudsLeftBase64 =
      "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";
  String _cloudsRightBase64 =
      "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";

  late Uint8List _airplaneBytes;
  late Uint8List _cloudsCenterBytes;
  late Uint8List _cloudsLeftBytes;
  late Uint8List _cloudsRightBytes;

  // 云朵漂浮值
  int _cloudFloatValue = 0;

  // 漂浮计时器
  Timer? _cloudFloatTimer;

  // 是否漂浮
  bool _isCloudFloat = false;

  set isCloudFloat(bool value) {
    if (_isCloudFloat != value) {
      _isCloudFloat = value;
      if (_isCloudFloat) {
        _cloudFloatValue = 0;
        _cloudFloat();
      }
    }
  }

  // 刷新结束
  bool _isRefreshed = false;

  @override
  void initState() {
    super.initState();
    _airplaneBytes = base64Decode(_airplaneBase64);
    _cloudsCenterBytes = base64Decode(_cloudsCenterBase64);
    _cloudsLeftBytes = base64Decode(_cloudsLeftBase64);
    _cloudsRightBytes = base64Decode(_cloudsRightBase64);
  }

  @override
  void dispose() {
    if (_cloudFloatTimer != null && _cloudFloatTimer!.isActive) {
      _cloudFloatTimer!.cancel();
    }
    super.dispose();
  }

  // 云朵漂浮
  void _cloudFloat() {
    if (!_isCloudFloat &&
        _cloudFloatValue ~/ 20 % 2 == 0 &&
        _cloudFloatValue % 20 == 0) {
      _cloudFloatValue = 0;
      _cloudFloatTimer!.cancel();
      return;
    }
    _cloudFloatTimer = Timer(Duration(milliseconds: 40), () {
      setState(() {
        _cloudFloatValue++;
        _cloudFloat();
      });
    });
  }

  @override
  Widget build(BuildContext context) {
    if (_refreshState == RefreshMode.armed ||
        _refreshState == RefreshMode.refresh) {
      isCloudFloat = true;
    } else if (_refreshState == RefreshMode.refreshed) {
      isCloudFloat = false;
      _isRefreshed = true;
    } else if (_refreshState == RefreshMode.inactive) {
      isCloudFloat = false;
      _isRefreshed = false;
    }
    // 计算云朵漂浮系数
    double _cloudFloatCoefficient;
    if (_cloudFloatValue == 0) {
      _cloudFloatCoefficient = 0;
    } else {
      if (_cloudFloatValue ~/ 20 % 2 == 0) {
        _cloudFloatCoefficient = _cloudFloatValue % 20 / 10;
      } else {
        _cloudFloatCoefficient = 2 - _cloudFloatValue % 20 / 10;
      }
    }
    // 中间云朵大小
    double centerCloudHeight;
    if (_pulledExtent > _indicatorExtent) {
      centerCloudHeight = 38.0 +
          (_pulledExtent - _indicatorExtent) / 5 -
          _cloudFloatCoefficient;
    } else {
      centerCloudHeight = 38.0 - _cloudFloatCoefficient;
    }
    // 飞机位置
    double airplaneBottom;
    double airplaneLeft;
    if (_isRefreshed && _pulledExtent < _indicatorExtent) {
      airplaneBottom = _indicatorExtent / 2 +
          (_indicatorExtent - _pulledExtent) * 22.5 / _indicatorExtent * 5 +
          _cloudFloatCoefficient;
      airplaneLeft = 122.5 +
          (_indicatorExtent - _pulledExtent) * 122.5 / _indicatorExtent * 3;
    } else {
      airplaneBottom = _pulledExtent < _indicatorExtent
          ? _pulledExtent / 2 + _cloudFloatCoefficient
          : _indicatorExtent / 2 + _cloudFloatCoefficient;
      airplaneLeft = _pulledExtent < _indicatorExtent
          ? _pulledExtent * 122.5 / _indicatorExtent
          : 122.5;
    }
    return Stack(
      children: <Widget>[
        // Image.asset(
        //   'assets/home/top_bg_icon.png',
        //   width: 1920,
        // ),
        //蓝天(背景)
        Container(
          width: double.infinity,
          color: widget.backgroundColor,
        ),
        // 左边云朵
        Positioned(
          bottom: -2.0,
          left: 0.0,
          child: Container(
            height: 38.0 + _cloudFloatCoefficient,
            width: 120.0,
            child: Image.memory(
              _cloudsLeftBytes,
              fit: BoxFit.fitHeight,
            ),
          ),
        ),
        // 右边云朵
        Positioned(
          bottom: -2.0,
          right: 0.0,
          child: Container(
            height: 38.0 + _cloudFloatCoefficient,
            width: 120.0,
            child: Image.memory(
              _cloudsRightBytes,
              fit: BoxFit.fitHeight,
            ),
          ),
        ),
        // 飞机
        Positioned(
          top: 0.0,
          left: 0.0,
          right: 0.0,
          child: Align(
            alignment: Alignment.topCenter,
            child: Container(
              width: 300.0,
              height: _indicatorExtent,
              child: Stack(
                children: <Widget>[
                  Positioned(
                    bottom: airplaneBottom,
                    left: airplaneLeft,
                    child: Container(
                      height: 17.5,
                      width: 55.0,
                      child: Image.memory(
                        _airplaneBytes,
                        fit: BoxFit.fill,
                      ),
                    ),
                  ),
                ],
              ),
            ),
          ),
        ),
        // 中间云朵
        Positioned(
          bottom: -2.0,
          left: 0.0,
          right: 0.0,
          child: Align(
            alignment: Alignment.bottomCenter,
            child: Container(
              height: centerCloudHeight,
              child: Image.memory(
                _cloudsCenterBytes,
                fit: BoxFit.fitHeight,
              ),
            ),
          ),
        ),
      ],
    );
  }
}
